/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import com.sap.cds.Struct;

public class ClassMethods {

	private final Map<String, List<Method>> classMethods = new HashMap<>();

	private final static Map<Class<?>, ClassMethods> createdClassMethods = new ConcurrentHashMap<>();

	public static ClassMethods create(Class<?> clazz) {

		ClassMethods cachedClassMethods = createdClassMethods.get(clazz);
		if (cachedClassMethods == null) {
			cachedClassMethods = new ClassMethods(clazz);
			createdClassMethods.put(clazz, cachedClassMethods); // concurrent write is ok. safes a synchronizer.
		}
		return cachedClassMethods;
	}

	private ClassMethods(Class<?> clazz) {
		for(Method m : clazz.getMethods()) {
			List<Method> originalMethods = classMethods.get(m.getName());
			if (originalMethods == null) {
				originalMethods = new ArrayList<>();
				classMethods.put(m.getName(), originalMethods);
			}
			originalMethods.add(m);
		}
	}

	public Method lookupMethod(Method proxyMethod) {

		List<Method> originalMethods = classMethods.get(proxyMethod.getName());
		if (originalMethods != null) {
			for (Method originalMethod : originalMethods) {
				if (originalMethod.getParameterCount() == proxyMethod.getParameterCount()) {

					Class<?>[] originalParameterTypes = originalMethod.getParameterTypes();
					Class<?>[] proxyParameterTypes = proxyMethod.getParameterTypes();

					boolean sameParameterList = true;
					for (int i = 0; i < originalMethod.getParameterCount(); ++i) {
						if (originalParameterTypes[i] != proxyParameterTypes[i] ) {
							sameParameterList = false;
							break;

						}
					}
					if (sameParameterList) {
						return originalMethod;
					}
				}
			}
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	public static <V, T extends V> T as(Class<T> targetClazz, Class<V> baseClazz, V impl, Supplier<Map<String, Object>> additionalData) {

		if(targetClazz.isAssignableFrom(impl.getClass())) {
			return (T) impl;
		}

		final ClassMethods classMethods = ClassMethods.create(baseClazz);
		T mapAccessor = Struct.access(additionalData.get()).as(targetClazz);

		return (T) Proxy.newProxyInstance(targetClazz.getClassLoader(), new Class[] { targetClazz, baseClazz },
				(proxy, method, methodArgs) -> {
					// if the method exists on the generic UserInfo, invoke directly on this object
					Method originalMethod = classMethods.lookupMethod(method);
					if(originalMethod != null) {
						// prevent creating further proxies
						if(method.getName().equals("as") && methodArgs.length == 1 && ((Class<?>) methodArgs[0]).isAssignableFrom(targetClazz)) {
							return proxy;
						}
						return originalMethod.invoke(impl, methodArgs);
					}

					// method only exists on the interface
					return method.invoke(mapAccessor, methodArgs);
				});
	}
}
