/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import java.util.Locale;
import java.util.Objects;

import org.slf4j.helpers.MessageFormatter;

import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.ServiceException;

public class ErrorStatusException extends ServiceException {

	private static final long serialVersionUID = 1L;

	public ErrorStatusException(ErrorStatus errorStatus, Object... args) {
		// the error status code is the message key used for translation
		super(errorStatus, errorStatus.getCodeString(), args);
	}

	@Override
	public String getMessage() {
		// ensure we always log the default error status description
		return getDefaultErrorStatusDescription();
	}

	@Override
	public String getLocalizedMessage(Locale locale) {
		String localized = super.getLocalizedMessage(locale);
		// no translation for error status found, therefore we use the default description
		if(Objects.equals(localized, getPlainMessage())) {
			// always fallback to the HTTP error text if desired
			boolean stackMessages = Utils.getErrorsProperties().getStackMessages().isEnabled();
			if (errorStatus instanceof CdsErrorStatuses statuses && !stackMessages) {
				return new ErrorStatusException(statuses.getHttpError()).getLocalizedMessage(locale);
			} else if (!(errorStatus instanceof ErrorStatuses) && !stackMessages) {
				return new ErrorStatusException(ErrorStatuses.SERVER_ERROR).getLocalizedMessage(locale);
			}
			return getDefaultErrorStatusDescription();
		}
		return localized;
	}

	/**
	 * Returns the default description of the error status
	 */
	private String getDefaultErrorStatusDescription() {
		return MessageFormatter.arrayFormat(getErrorStatus().getDescription(), args).getMessage();
	}

}
