/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import java.util.Optional;

import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.util.CqnStatementUtils;
import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.trace.SpanBuilder;
import io.opentelemetry.api.trace.SpanId;
import io.opentelemetry.api.trace.SpanKind;
import io.opentelemetry.api.trace.TraceId;
import io.opentelemetry.context.Context;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.request.RequestContext;

import io.opentelemetry.api.GlobalOpenTelemetry;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.StatusCode;
import io.opentelemetry.api.trace.Tracer;
import io.opentelemetry.api.trace.TracerProvider;
import org.slf4j.event.Level;

public class OpenTelemetryUtils {

	// Span Attributes
	public static final AttributeKey<String> CDS_TENANT = AttributeKey.stringKey("cds.tenant");
	public static final AttributeKey<String> CDS_SERVICE = AttributeKey.stringKey("cds.service");
	public static final AttributeKey<String> CDS_EVENT_NAME = AttributeKey.stringKey("cds.event_name");
	public static final AttributeKey<String> CDS_ENTITY_NAME = AttributeKey.stringKey("cds.entity_name");
	public static final AttributeKey<String> CDS_OUTBOX_TARGET = AttributeKey.stringKey("cds.outbox.target");
	public static final AttributeKey<String> CDS_CQN_STATEMENT = AttributeKey.stringKey("cds.cqn.statement");
	public static final AttributeKey<String> CDS_CQN_TARGET_ENTITY = AttributeKey.stringKey("cds.cqn.target_entity");
	public static final AttributeKey<String> CDS_CQN_TO_PROTOCOL = AttributeKey.stringKey("cds.cqn.to_protocol");
	public static final AttributeKey<Boolean> CDS_ODATA_IS_BATCH = AttributeKey.booleanKey("cds.odata.is_batch");
	public static final AttributeKey<Boolean> CDS_ODATA_IS_CHANGESET = AttributeKey.booleanKey("cds.odata.is_changeset");
	public static final AttributeKey<Long> HTTP_STATUS_CODE = AttributeKey.longKey("http.status_code");
	public static final AttributeKey<String> HTTP_METHOD = AttributeKey.stringKey("http.method");
	public static final AttributeKey<String> HTTP_SCHEME = AttributeKey.stringKey("http.scheme");
	public static final AttributeKey<String> HTTP_TARGET = AttributeKey.stringKey("http.target");

	private final static String CDS_INSTRUMENTATION_SCOPE = "com.sap.cds";
	private static Tracer tracer = GlobalOpenTelemetry.get().getTracer(CDS_INSTRUMENTATION_SCOPE);

	public enum CdsSpanType {
		REQUEST_CONTEXT (Level.DEBUG, "RequestContext"),
		CHANGESET_CONTEXT (Level.DEBUG, "ChangeSetContext"),
		EMIT (Level.DEBUG, "Emit"),
		ODATA_BATCH (Level.INFO, "ODataBatch"),
		CQN (Level.INFO, "CQN"),
		OUTBOX (Level.INFO, "OutboxCollector"),
		DRAFT_GC (Level.INFO, "DraftGarbageCollection");

		private final Level logLevel;
		private final Logger logger;

		CdsSpanType(Level logLevel, String localName) {
			this.logLevel = logLevel;
			this.logger = LoggerFactory.getLogger("com.sap.cds.otel.span." + localName);
		}

		public Logger getLogger() { return this.logger; }

		public boolean isEnabledForLogging() {
			return this.logger.isEnabledForLevel(this.logLevel);
		}
	}

	private OpenTelemetryUtils() {
		// hidden
	}

	public static Optional<Span> createSpan(CdsSpanType type) {
		return createSpan(type, null, SpanKind.INTERNAL);
	}

	public static Optional<Span> createSpan(CdsSpanType type, SpanKind kind) {
		return createSpan(type, null, kind);
	}

	public static Optional<Span> createSpan(CdsSpanType type, Context parentContext) {
		return createSpan(type, parentContext, SpanKind.INTERNAL);
	}

	public static Optional<Span> createSpan(CdsSpanType type, Context parentContext, SpanKind kind) {
		if(type.isEnabledForLogging()) {
			SpanBuilder spanBuilder =  tracer.spanBuilder(type.toString()).setSpanKind(kind);

			/*
			 * For Java Executors, auto instrumentation might be in place. So, if there is a valid trace
			 * context available, we treat it with precedence over the manual propagation and will be
			 * automatically propagated. No need for us to set the parent manually.
			 */
			if(!isValidContext(Context.current()) && parentContext != null) {
				spanBuilder.setParent(parentContext);
			}

			return Optional.of(spanBuilder.startSpan());
		}
		return Optional.empty();
	}

	private static boolean isValidContext(Context ctx) {
		Span span = Span.fromContext(ctx);
		return TraceId.isValid(span.getSpanContext().getTraceId()) && SpanId.isValid(span.getSpanContext().getSpanId());
	}
  
	public static void updateSpan(Optional<Span> span, String serviceName, String eventName, String entityName) {
		span.ifPresent(s -> {
			s.updateName(serviceName + " (" + eventName + ")");
			s.setAttribute(CDS_SERVICE, String.valueOf(serviceName));
			s.setAttribute(CDS_EVENT_NAME, String.valueOf(eventName));
			s.setAttribute(CDS_ENTITY_NAME, String.valueOf(entityName));
		});
	}

	public static void updateSpan(Optional<Span> span, String requestContextId, RequestContext requestContext) {
		span.ifPresent(s -> {
			s.updateName("RequestContext " + requestContextId);
			s.setAttribute("cds.tenant", String.valueOf(requestContext.getUserInfo().getTenant()));
			s.setAttribute("cds.systemUser", String.valueOf(requestContext.getUserInfo().isSystemUser()));
			s.setAttribute("cds.internalUser", String.valueOf(requestContext.getUserInfo().isInternalUser()));
			s.setAttribute("cds.privilegedUser", String.valueOf(requestContext.getUserInfo().isPrivileged()));
			s.setAttribute("cds.locale", String.valueOf(requestContext.getParameterInfo().getLocale()));
			s.setAttribute("cds.validFrom", String.valueOf(requestContext.getParameterInfo().getValidFrom()));
			s.setAttribute("cds.validTo", String.valueOf(requestContext.getParameterInfo().getValidTo()));
		});
	}

	public static void updateSpan(Optional<Span> span, ChangeSetContext changeSetContext) {
		span.ifPresent(s -> {
			s.updateName("ChangeSetContext " + changeSetContext.getId());
		});
	}

	public static void updateSpan(Optional<Span> span, CdsRuntime runtime, String cqnOperation, CdsEntity targetEntity, CqnStatement cqnStatement, String targetProtocol) {
		span.ifPresent(s -> {
			boolean logValues = runtime.getEnvironment().getCdsProperties().getSecurity().isLogPotentiallySensitive();

			if (cqnStatement != null && CdsSpanType.CQN.logger.isDebugEnabled()) {
				// for performance reasons, the statement gets logged only in DEBUG level
				String statement = (logValues) ? cqnStatement.toJson() : CqnStatementUtils.anonymizeStatement(cqnStatement).toJson();
				s.setAttribute(CDS_CQN_STATEMENT, statement);
			}
			StringBuilder builder = new StringBuilder();
			builder.append("CQN " + cqnOperation);
			if (targetEntity != null) {
				builder.append(" " + targetEntity.getName());
				s.setAttribute(CDS_CQN_TARGET_ENTITY, targetEntity.getQualifiedName());
			}

			s.setAttribute(CDS_CQN_TO_PROTOCOL, targetProtocol);
			s.updateName(builder.toString());
		});
	}

	public static void endSpan(Optional<Span> span) {
		span.ifPresent(Span::end);
	}

	public static void recordException(Optional<Span> span, Exception e) {
		span.ifPresent(s -> {
			s.recordException(e);
			s.setStatus(StatusCode.ERROR);
		});
	}

	@VisibleForTesting
	public static void setTracerProvider(TracerProvider tracerProvider) {
		tracer = tracerProvider.tracerBuilder(CDS_INSTRUMENTATION_SCOPE).setInstrumentationVersion("1.0.0").build();
	}
}
