package com.sap.cds.services.utils;

/**
 * In certain situations, it might be required to suppress the creation of new
 * Request Contexts. Especially when performing
 * outbound http requests (e.g. to sidecar) nduring the creation or
 * initialization of a Request Context, this causes endless
 * recurring creation of new Request Contexts in combination with resilience
 * patterns (e.g. CdsThreadContextDecorator comes
 * in the picture) which delegates execution to dedicated threads.
 *
 * In such situations, suppressing the creation of new Request Contexts will
 * allow the outbound http request to be executed
 * in the resilience thread without any Request Context. However, this will only
 * work if the outbound http requests
 * are not dependent on a tenant context taken from the current Request Context
 * (ie. OnBehalf.TECHNICAL_PROVIDER_TENANT).
 */
public class RequestContextSuppressor implements AutoCloseable {

	private static final ThreadLocal<Boolean> suppressNewRequestContext = ThreadLocal.withInitial(() -> Boolean.FALSE);

	public RequestContextSuppressor() {
		suppressNewRequestContext.set(Boolean.TRUE);
	}

	public static boolean isSuppressNewRequestContext() {
		return suppressNewRequestContext.get();
	}

	@Override
	public void close() {
		suppressNewRequestContext.remove();
	}
}
