/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.destination;

import java.net.URI;
import java.util.Collections;

import com.sap.cds.services.ServiceException;
import com.sap.cds.services.utils.environment.ServiceBindingUtils;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultOAuth2PropertySupplier;
import com.sap.cloud.sdk.cloudplatform.connectivity.OAuth2ServiceBindingDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationOptions;

import io.vavr.control.Option;

/**
 * Dedicated {@link com.sap.cloud.sdk.cloudplatform.connectivity.OAuth2PropertySupplier} which parses service bindings of
 * type <code>xsuaa</code>. As these bindings only provide credentials, but no service url, the service url can be dynamically provided
 * via the class {@link UrlOptions}. If a caller does not provide a {@link UrlOptions}, an exception will be thrown.
 */
public class XsuaaOAuth2PropertySupplier extends DefaultOAuth2PropertySupplier {

	private static boolean initialized = false;

	public static synchronized void initialize() {
		if (!initialized) {
			OAuth2ServiceBindingDestinationLoader.registerPropertySupplier(
					options -> ServiceBindingUtils.matches(options.getServiceBinding(), "xsuaa"),
					XsuaaOAuth2PropertySupplier::new);
			initialized = true;
		}
	}

	public XsuaaOAuth2PropertySupplier(ServiceBindingDestinationOptions options) {
		super(options, Collections.emptyList());
	}

	@Override
	public URI getServiceUri() {
		Option<String> maybeOption = options.getOption(UrlOptions.class);
		if (maybeOption.isEmpty()) {
			throw new ServiceException("No `URL` has been provided.");
		}

		String url = maybeOption.get();
		return URI.create(url);
	}

	public static final class UrlOptions implements ServiceBindingDestinationOptions.OptionsEnhancer<String> {

		private String url;

		public UrlOptions(String url) {
			this.url = url;
		}

		@Override
		public String getValue() {
			return url;
		}
	}
}
