/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.model;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsDefinitionNotFoundException;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsElementNotFoundException;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class CdsModelUtils {

	public static final String $USER = "$user";

	private final static ObjectMapper mapper = new ObjectMapper()
			.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
			.configure(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY, true);

	private final CdsRuntime runtime;

	public CdsModelUtils(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	/**
	 * Returns {@code true} if the passed model element (either service or entity) has no effective restriction.
	 * Please note that this does not include hierarchical dependencies.
	 *
	 * @param cdsModelElement	The element from the CDS model
	 * @return	{@code true} if public
	 */
	public Optional<Boolean> isPublic(CdsAnnotatable cdsModelElement) {
		String mode = runtime.getEnvironment().getCdsProperties().getSecurity().getAuthentication().getMode().trim();
		if(mode.equalsIgnoreCase("always")) {
			return Optional.of(false);
		} else if (mode.equalsIgnoreCase("never")) {
			return Optional.of(true);
		}

		// model element is public if
		// - there is a restriction containing a 'wildcard all' privilege
		// - there is no restriction and authentication mode is 'model-relaxed'
		Restriction restriction = CdsModelUtils.getRestrictionOrNull(cdsModelElement);
		if (restriction != null) {
			return Optional.of(restriction.privileges().anyMatch(priv -> 
					priv.getGrants().contains(Privilege.PredefinedGrant.ALL.toString()) &&
					priv.getRoles().contains(Privilege.PredefinedRole.ANY_USER.toString()) && !priv.hasWhereUsing($USER)));
		} else if (cdsModelElement instanceof CdsService) {
			return Optional.of(mode.equalsIgnoreCase("model-relaxed"));
		} else {
			return Optional.empty(); // inherit
		}
	}

	/**
	 * Returns a list of all events that are not restricted for the given model element, although the element itself might be restricted in general.
	 *
	 * @param cdsModelElement	The CDS model element
	 * @return	The list of public events
	 */
	public List<String> getPublicEvents(CdsAnnotatable cdsModelElement) {

		Set<String> publicEvents = new HashSet<>();
		Restriction restriction = CdsModelUtils.getRestrictionOrNull(cdsModelElement);
		if (restriction != null) {
			publicEvents = restriction.privileges().filter(priv -> {
				return priv.getRoles().contains(Privilege.PredefinedRole.ANY_USER.toString()) && !priv.hasWhereUsing($USER);
			}).flatMap(priv -> priv.getGrants().stream()).collect(Collectors.toSet());
		}
		if (publicEvents.contains(Privilege.PredefinedGrant.ALL.toString())) {
			publicEvents = Collections.singleton(Privilege.PredefinedGrant.ALL.toString()); // '*' dominates all others
		}
		return publicEvents.stream().collect(Collectors.toList());
	}


	/**
	 * Returns a {@link Restriction} instance reflecting @restrict resp. @requires on the given service or {@code null}
	 * if not annotated.
	 *
	 * @param cdsModelElement	The CDS model element to be inspected
	 * @return	The {@link Restriction} instance
	 */
	public static Restriction getRestrictionOrNull(CdsAnnotatable cdsModelElement) {

		Restriction result = readPrivileges(cdsModelElement);

		Object annotationRoles = CdsAnnotations.REQUIRES.getOrDefault(cdsModelElement);
		if (annotationRoles != null) {
			if (result == null) {
				result = new Restriction();
			}
			String[] roles = mapper.convertValue(annotationRoles, String[].class);
			for (String role : roles) {
				result.addPrivilege( new Privilege().addGrant(Privilege.PredefinedGrant.ALL).addRole(role) );
			}
		}

		return result; // could be null if there are no privileges
	}

	@SuppressWarnings("unchecked")
	private static Restriction readPrivileges(CdsAnnotatable cdsModelElement) {
		List<Object> privs = CdsAnnotations.RESTRICT.getOrDefault(cdsModelElement);
		CdsStructuredType type = cdsModelElement instanceof CdsStructuredType t ? t : null;
		if (privs != null) {
			Privilege[] privileges = mapper.convertValue(privs, Privilege[].class);
			for (int i = 0; i < privileges.length; i++) {
			    privileges[i].readWhere(type, (Map<String, Object>) privs.get(i));
			}
			return new Restriction().addPrivileges(privileges);
		}
		return null;
	}

	public static AnalysisResult getEntityPath(CqnStructuredTypeRef ref, CdsModel model) {
		try {
			return CqnAnalyzer.create(model).analyze(ref);
		} catch (CdsElementNotFoundException e) {
			throw new ErrorStatusException(CdsErrorStatuses.ELEMENT_NOT_FOUND, e.getElementName(),
					e.getDefinition().getName(), e);
		} catch (CdsDefinitionNotFoundException e) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_FOUND, e.getQualifiedName(), e);
		}
	}

	public static boolean isChildEntity(CqnStructuredTypeRef ref, CdsModel model) {
		CdsEntity entity = null;
		for(Segment s: ref.segments()) {
			if (entity == null) {
				entity = model.getEntity(s.id());
			} else {
				String name = entity.getQualifiedName();
				CdsElement association = entity.findAssociation(s.id())
						.orElseThrow(() -> new ErrorStatusException(CdsErrorStatuses.ASSOCIATION_NOT_FOUND, s.id(), name));
				if (!association.getType().as(CdsAssociationType.class).isComposition()) {
					return false;
				}
			}
		}
		return true;
	}

	public static CdsEntity getTargetEntity(CqnSelect select, CdsModel model) {
		return getEntityPath(select, model).targetEntity();
	}

	public static CdsEntity getTargetEntity(CqnStructuredTypeRef ref, CdsModel model) {
		return getEntityPath(ref, model).targetEntity();
	}

	public static AnalysisResult getEntityPath(CqnSelect select, CdsModel model) {
		return getEntityPath(select.ref(), model);
	}

	public static CdsEntity getRefTarget(CqnStructuredTypeRef ref, CdsEntity target) {
		CdsEntity refTarget = target;
		for(Segment s : ref.segments()) {
			if(s.id().equals("*")) {
				return null; // can't handle *
			}
			refTarget = refTarget.getTargetOf(s.id());
		}
		return refTarget;
	}

	public static String getTargetKeysAsString(CdsModel model, CqnStatement statement) {
		AnalysisResult analysis = CqnAnalyzer.create(model).analyze(statement.ref());
		String keyValues = analysis.targetKeyValues().entrySet().stream() //
				.map(entry -> entry.getKey() + "=" + entry.getValue()) //
				.collect(Collectors.joining(","));
		return keyValues;
	}

	public static Optional<CdsFunction> findFunction(CdsService service, CdsEntity entity, String name) {
		Optional<CdsFunction> result = service.functions().filter(f -> f.getName().equals(name)).findAny();
		if (entity != null && result.isEmpty()) {
			return entity.functions().filter(f -> f.getName().equals(name)).findAny();
		}
		return result;
	}

	public static Optional<CdsAction> findAction(CdsService service, CdsEntity entity, String name) {
		Optional<CdsAction> result = service.actions().filter(a -> a.getName().equals(name)).findAny();
		if (entity != null && result.isEmpty()) {
			return entity.actions().filter(a -> a.getName().equals(name)).findAny();
		}
		return result;
	}

}
