/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.model;

import java.lang.reflect.Method;
import java.time.Duration;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.mtx.impl.CacheParams;
import com.sap.cds.mtx.impl.MetaDataAccessorImpl;
import com.sap.cds.mtx.impl.MetaDataAccessorImpl.CdsModelCreator;
import com.sap.cds.mtx.impl.MetaDataAccessorImpl.EdmxModelCreator;
import com.sap.cds.mtx.impl.MetaDataAccessorImpl.I18nResourceCreator;
import com.sap.cds.mtx.impl.ModelProviderAccess;
import com.sap.cds.mtx.impl.SidecarAccess;
import com.sap.cds.mtx.impl.SidecarAccessV1;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.environment.CdsProperties.Model.Provider;
import com.sap.cds.services.environment.CdsProperties.MultiTenancy.Sidecar;
import com.sap.cds.services.environment.CdsProperties.MultiTenancy.Sidecar.Cache;
import com.sap.cds.services.mt.ExtensibilityService;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.XsuaaUtils;
import com.sap.cds.services.utils.destination.XsuaaOAuth2PropertySupplier;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.mt.tools.api.ResilienceConfig;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultHttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationProperty;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.OnBehalfOf;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationOptions;

import io.vavr.control.Option;

public class DynamicModelUtils {

	static {
		XsuaaOAuth2PropertySupplier.initialize();
	}

	private static final Logger logger = LoggerFactory.getLogger(DynamicModelUtils.class);

	private final CdsRuntime runtime;
	private final Provider providerConfig;
	private final Sidecar sidecarConfig;
	private final XsuaaUtils xsuaaUtils;
	private final boolean mtxs;

	public DynamicModelUtils(CdsRuntime runtime) {
		this.runtime = runtime;
		this.providerConfig = runtime.getEnvironment().getCdsProperties().getModel().getProvider();
		this.sidecarConfig = runtime.getEnvironment().getCdsProperties().getMultiTenancy().getSidecar();
		this.mtxs = runtime.getEnvironment().getCdsProperties().getMultiTenancy().getMtxs().isEnabled();
		this.xsuaaUtils = new XsuaaUtils(runtime);
	}

	/**
	 * Returns true, if the static model, packaged with the application, can be loaded
	 * Retrieves the required information from the current {@link RequestContext}
	 *
	 * @return true, if the static model, packaged with the application, can be loaded
	 */
	public boolean useStaticModel() {
		RequestContext requestContext = RequestContext.getCurrent(runtime);
		return useStaticModel(requestContext.getUserInfo(), requestContext.getFeatureTogglesInfo());
	}

	/**
	 * Returns true, if the static model, packaged with the application, can be loaded
	 *
	 * @param userInfo the {@link UserInfo}
	 * @param featureTogglesInfo the {@link FeatureTogglesInfo}
	 * @return true, if the static model, packaged with the application, can be loaded
	 */
	public boolean useStaticModel(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
		boolean hasAllToggles = featureTogglesInfo.getEnabledFeatureToggles().anyMatch(t -> t.getName().trim().equals("*"));
		return userInfo.getTenant() == null && (!isModelProviderEnabled() || hasAllToggles);
	}

	/**
	 * @return {@code true} if ModelProviderService is configured
	 */
	public boolean isModelProviderEnabled() {
		return !StringUtils.isEmpty(providerConfig.getUrl()) ||
				(mtxs && !StringUtils.isEmpty(sidecarConfig.getUrl()));
	}

	public String getModelProviderUrl() {
		return !StringUtils.isEmpty(providerConfig.getUrl()) ? providerConfig.getUrl() : sidecarConfig.getUrl();
	}

	/**
	 * @return {@code true} if Classic MT Sidecar is configured
	 */
	public boolean isClassicSidecarEnabled() {
		return !mtxs &&	!StringUtils.isEmpty(sidecarConfig.getUrl());
	}

	/**
	 * Indicates whether the model is provided by a external service (i.e. it's dynamic) or not.
	 * Dynamic models can be served by:
	 * - classic sidecar
	 * - ModelProvider service (either as mtxs service or standalone)
	 *
	 * @return {@code true} if the model is dynamic
	 */
	public boolean isDynamicModelEnabled() {
		return isClassicSidecarEnabled() || isModelProviderEnabled();
	}

	public <T> MetaDataAccessor<T> createMetadataAccessor(EdmxModelCreator<T> strToEdmx, CdsModelCreator strToModel, I18nResourceCreator strToI18n) {
		// create accessor
		MetaDataAccessor<T> accessor = buildMetadataAccessor(strToEdmx, strToModel, strToI18n);

		// register accessor with extensibility service
		if(accessor != null) {
			ExtensibilityService extService = runtime.getServiceCatalog().getService(ExtensibilityService.class, ExtensibilityService.DEFAULT_NAME);
			if(extService != null) {
				extService.on(ExtensibilityService.EVENT_MODEL_CHANGED, null, context -> {
					String tenant = context.getUserInfo().getTenant();
					if (sidecarConfig.getCache().isRefreshEager()) {
						accessor.refresh(tenant);
					} else {
						accessor.evict(tenant);
					}
				});
			}
		}

		return accessor;
	}

	private <T> MetaDataAccessor<T> buildMetadataAccessor(EdmxModelCreator<T> strToEdmx, CdsModelCreator strToModel, I18nResourceCreator strToI18n) {
		SidecarAccess sidecarAccess = null;
		CacheParams cacheParams = null;

		if (isModelProviderEnabled()) {
			sidecarAccess = new ModelProviderAccess(request -> {}, getResilienceConfig());
			cacheParams = getCacheParams(providerConfig.getCache());
		} else if (isClassicSidecarEnabled()) {
			cacheParams = getCacheParams(sidecarConfig.getCache());
			sidecarAccess = new SidecarAccessV1(request -> {}, getResilienceConfig());
		}

		if (sidecarAccess != null && cacheParams != null) {
			return new MetaDataAccessorImpl<>(new MetaDataAccessorImpl.MetaDataAccessorConfig.Builder()
					.sidecarAccess(sidecarAccess).cacheParams(cacheParams)
					.strToEdmx(strToEdmx).strToModel(strToModel).strToI18n(strToI18n)
					.build(), null);
		}
		return null;
	}

	private CacheParams getCacheParams(Cache cache) {
		long maximumSize = cache.getMaxSize();
		long expirationDuration = cache.getExpirationTime();
		long refreshDuration = cache.getRefreshTime();
		return new CacheParams(maximumSize, Duration.ofSeconds(expirationDuration), Duration.ofSeconds(refreshDuration), false, false);
	}

	public ResilienceConfig getResilienceConfig() {
		return ResilienceConfig.builder().numOfRetries(3).retryInterval(Duration.ofMillis(500)).build();
	}

	public HttpDestination createSidecarDestination(String destinationName, String targetUrl) {
		List<ServiceBinding> uaaBindings = xsuaaUtils.getXsuaaServiceBindings();
		ServiceBindingDestinationOptions.OptionsEnhancer<?> enhancer = null;
		ServiceBinding binding;

		if (!uaaBindings.isEmpty()) {
			binding = uaaBindings.get(0);
			enhancer = new XsuaaOAuth2PropertySupplier.UrlOptions(targetUrl);
		} else if (!xsuaaUtils.getIasServiceBindings().isEmpty()) {
			binding = xsuaaUtils.getIasServiceBindings().get(0);
			enhancer = proxyOptionsEnhancer(targetUrl);
		} else {
			logger.debug("Initializing MTX sidecar destination '{}' without service binding.", destinationName);
			return DefaultHttpDestination.builder(targetUrl).name(destinationName).build();
		}

		logger.debug("Initializing MTX sidecar destination '{}' using service binding '{}'.", destinationName, binding.getName().get());

		DefaultHttpDestination http = (DefaultHttpDestination) ServiceBindingDestinationLoader.defaultLoaderChain().
				getDestination(ServiceBindingDestinationOptions.forService(binding).
						withOption(enhancer).
						onBehalfOf(OnBehalfOf.TECHNICAL_USER_PROVIDER).build());

		return proxyDestinationWithOverridenName(http, destinationName);
	}

	// TODO: remove once Cloud SDK 5 is mandatory
	private HttpDestination proxyDestinationWithOverridenName(DefaultHttpDestination http, String destinationName) {
		return (HttpDestination) java.lang.reflect.Proxy.newProxyInstance(
			http.getClass().getClassLoader(),
			new Class[] { HttpDestination.class },
			(proxy, method, args) -> {
				if (method.getName().equals("get") && args.length == 1 &&
							(args[0].equals(DestinationProperty.NAME) || args[0].equals("Name"))) {
					return Option.of(destinationName);
				}
				return method.invoke(http, args);
			});
	}

	// TODO: remove once Cloud SDK 5 is mandatory
	private static ServiceBindingDestinationOptions.OptionsEnhancer<?> proxyOptionsEnhancer(String targetUrl) {
		try {
			Class<?> clazz = Class.forName("com.sap.cloud.sdk.cloudplatform.connectivity.BtpServiceOptions$IasOptions");
			Method method = clazz.getMethod("withTargetUri", String.class);
			return (ServiceBindingDestinationOptions.OptionsEnhancer<?>)method.invoke(null, targetUrl);
		} catch (Exception e) {
			throw new ServiceException("Service bindings of type 'identity' are only supported with Cloud SDK 5");
		}
	}
}
