/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ErrorStatuses;

/**
 * Implementation of the {@link ErrorStatus} interface, which provides error codes used by the stack
 */
public enum CdsErrorStatuses implements ErrorStatus {

	// 400
	INVALID_CQN(400001, "Invalid CQN: {}", ErrorStatuses.BAD_REQUEST), // TODO CDS4J should provide access to details (element, entity, etc.)
	INVALID_DATE_VALUE(400002, "Provided value in query parameter '{}' is not a valid date value (expected ISO 8601 date time format)", ErrorStatuses.BAD_REQUEST),
	ENTITY_NOT_DRAFT_ENABLED(400003, "Entity '{}' is not draft enabled", ErrorStatuses.BAD_REQUEST),
	ENTITY_NOT_ROOT(400004, "Entity '{}' is not the root entity of the draft-enabled composition tree", ErrorStatuses.BAD_REQUEST),
	ENTITY_NOT_ACTIVE(400005, "Only active draft-enabled entities (IsActiveEntity = true) are allowed", ErrorStatuses.BAD_REQUEST),
	ENTITY_NOT_INACTIVE(400006, "Only inactive draft-enabled entities (IsActiveEntity = false) are allowed", ErrorStatuses.BAD_REQUEST),
	INVALID_URI_RESOURCE(400007, "URI resource is invalid", ErrorStatuses.BAD_REQUEST),
	INVALID_PAYLOAD(400008, "Could not deserialize request payload: {}", ErrorStatuses.BAD_REQUEST), // TODO how to localize Olingo messages? mention OData?
	INVALID_EXPAND_STAR(400009, "Expand '*' can't be combined with further expands", ErrorStatuses.BAD_REQUEST),
	INVALID_SUBSTRING(400010, "Invalid number of parameters for filter function 'substring'", ErrorStatuses.BAD_REQUEST),
	MISSING_CONTENT_TYPE(400011, "'Content-Type' request header required", ErrorStatuses.BAD_REQUEST),
	VALUE_OUT_OF_RANGE(400012, "Value of element '{}' is out of range (minimum value '{}' and maximum value '{}')", ErrorStatuses.BAD_REQUEST),
	DRAFT_DEEP_UPDATE(400013, "Deep updates are not allowed when patching an inactive draft-enabled entity (IsActiveEntity = false)", ErrorStatuses.BAD_REQUEST),
	TENANT_NOT_KNOWN(400014, "Tenant '{}' not known", ErrorStatuses.BAD_REQUEST),
	INVALID_ENUM(400015, "Provided enum value of element '{}' is not in the list of allowed enum values '{}'", ErrorStatuses.BAD_REQUEST),
	INVALID_SORT_ELEMENT(400016, "Could not sort by element '{}' of entity '{}'", ErrorStatuses.BAD_REQUEST),
	INVALID_PARAMETERIZED_VIEW(400017, "You need to use the 'Set' navigation property when accessing a parameterized view", ErrorStatuses.BAD_REQUEST),
	INVALID_STRING_VALUE(400018, "Provided value of element '{}' does not match the expected pattern", ErrorStatuses.BAD_REQUEST),
	EXCEEDED_FIELD_LENGTH(400019, "Provided value of element '{}' exceeds maximum length of {}", ErrorStatuses.BAD_REQUEST),
	NO_CUSTOM_AGGREGATE_DEFINED(400020, "No custom aggregate is defined for element '{}'", ErrorStatuses.BAD_REQUEST),
	MISSING_ISACTIVEENTITY_KEY(400021, "The IsActiveEntity key is missing for path segments prior to an association to a draft enabled entity", ErrorStatuses.BAD_REQUEST),
	UNKONWN_AGGREGATION_METHOD(400022, "The aggregate method '{}' is unknown", ErrorStatuses.BAD_REQUEST),
	TARGET_ENTITY_MISSING(400023, "Foreign key value doesn't exist", ErrorStatuses.BAD_REQUEST),
	INVALID_CHARSET(400024, "Content-Type contains invalid charset", ErrorStatuses.BAD_REQUEST),
	UNSUPPORTED_CONTENT_TYPE(400025, "Unsupported content type '{}'", ErrorStatuses.BAD_REQUEST),
	UNSUPPORTED_PARAMETER_VALUE(400026, "Unsupported value '{}' of parameter '{}'", ErrorStatuses.BAD_REQUEST),
	MALFORMED_SKIPTOKEN(400027, "Value of $skiptoken is malformed", ErrorStatuses.BAD_REQUEST),
	BATCH_TOO_MANY_REQUESTS(400028, "Batch request contains too many requests", ErrorStatuses.BAD_REQUEST),
	INLINE_COUNT_ON_TO_ONE_ASSOC(400029, "Inline count is not permitted on to-one association '{}'", ErrorStatuses.BAD_REQUEST),

	// 401
	TENANT_CONTEXT_MISSING(401001, "Ensure authenticated user context is provided and contains tenant information", ErrorStatuses.UNAUTHORIZED),
	TOKEN_PARSING_FAILED(401002, "Failed to parse JWT token", ErrorStatuses.UNAUTHORIZED),
	EVENT_UNAUTHENTICATED(401003, "Not authenticated to send event '{}' to '{}'", ErrorStatuses.UNAUTHORIZED), // 401, "unauthorized" = unauthenticated

	// 403
	EVENT_FORBIDDEN(403001, "Not authorized to send event '{}' to '{}'", ErrorStatuses.FORBIDDEN), // 403, "forbidden" = unauthorized
	EVENT_FORBIDDEN_UNSUPPORTED_USER_ATTRIBUTES(403002, "Not authorized to send event '{}' to '{}'. Authorization requires a unique value of user attribute '{}'.", ErrorStatuses.FORBIDDEN),

	// 404
	SERVICE_NOT_FOUND(404001, "Service '{}' not found", ErrorStatuses.NOT_FOUND),
	ENTITY_NOT_FOUND(404002, "Entity '{}' not found", ErrorStatuses.NOT_FOUND),
	ACTION_NOT_FOUND(404003, "Action '{}' not found", ErrorStatuses.NOT_FOUND),
	FUNCTION_NOT_FOUND(404004, "Function '{}' not found", ErrorStatuses.NOT_FOUND),
	ASSOCIATION_NOT_FOUND(404005, "Association '{}' in entity '{}' not found", ErrorStatuses.NOT_FOUND),
	ELEMENT_NOT_FOUND(404006, "Element '{}' of entity '{}' not found", ErrorStatuses.NOT_FOUND),
	TENANT_NOT_FOUND(404007, "Tenant '{}' not found", ErrorStatuses.NOT_FOUND),
	JOB_NOT_FOUND(404008, "Asynchronous schema deployment job '{}' not found", ErrorStatuses.NOT_FOUND),
	ENTITY_INSTANCE_NOT_FOUND(404009, "Entity '{}' with key(s) '{}' not found", ErrorStatuses.NOT_FOUND),
	FUNCTION_RETURN_VALUE_NOT_FOUND(404010, "Function '{}' should return a single value, but no return value found", ErrorStatuses.NOT_FOUND),

	// 405
	ENTITY_NOT_READABLE(405001, "Entity '{}' is not readable", ErrorStatuses.METHOD_NOT_ALLOWED),
	ENTITY_NOT_INSERTABLE(405002, "Entity '{}' is not insertable", ErrorStatuses.METHOD_NOT_ALLOWED),
	ENTITY_NOT_UPDATABLE(405003, "Entity '{}' is not updatable", ErrorStatuses.METHOD_NOT_ALLOWED),
	ENTITY_NOT_DELETABLE(405004, "Entity '{}' is not deletable", ErrorStatuses.METHOD_NOT_ALLOWED),
	ENTITY_NOT_DELTA_UPDATABLE(405005, "Entity '{}' is not delta updatable", ErrorStatuses.METHOD_NOT_ALLOWED),

	// 406
	VALUE_ACCESS_NOT_ALLOWED(406001, "Not allowed to access $value of stream element '{}'", ErrorStatuses.NOT_ACCEPTABLE),

	// 409
	PARENT_NOT_EXISTING(409002, "Couldn't find parent entity for entity '{}'", ErrorStatuses.CONFLICT),
	VALUE_REQUIRED(409003, "Value of element '{}' in entity '{}' is required", ErrorStatuses.BAD_REQUEST), // initially was CONFLICT
	MULTIPLE_PARENTS(409004, "Found {} parent entities for entity '{}', but only a single one is allowed", ErrorStatuses.CONFLICT),
	MULTIPLE_SINGLETON_ENTRIES(409005, "Only a single entry is allowed, when upserting an existing singleton", ErrorStatuses.CONFLICT),
	UNIQUE_CONSTRAINT_VIOLATED(409006, "Unique constraint violated when inserting or updating entity '{}'", ErrorStatuses.CONFLICT),
	LOCK_TIMEOUT(409007, "Timeout while waiting for lock on entity '{}'", ErrorStatuses.CONFLICT),
	MISSING_VALUE(409008, "Value of element '{}' is required", ErrorStatuses.BAD_REQUEST),
	// 412
	ETAG_FAILED(412001, "ETag condition not met", ErrorStatuses.PRECONDITION_FAILED),

	// 421
	MISDIRECTED_ENTITY(421001, "Service '{}' does not contain entity '{}' and CQN statement can't be projected onto the service", ErrorStatuses.MISDIRECTED_REQUEST),

	// 428
	ETAG_REQUIRED(428001, "'If-Match' or 'If-None-Match' request header with etag condition required", ErrorStatuses.PRECONDITION_REQUIRED),
	ETAG_VALUE_INVALID(428002, "Invalid ETag: Value could not be converted to '{}'", ErrorStatuses.PRECONDITION_REQUIRED),

	// 500 (01: Event Handler)
	NO_ON_HANDLER(50001001, "No ON handler completed the processing", ErrorStatuses.SERVER_ERROR),
	HANDLER_FAILED(50001002, "Could not create handler instance of type '{}'", ErrorStatuses.SERVER_ERROR),
	HANDLER_NOT_ACCESSIBLE(50001003, "Can't access method handle for '{}'", ErrorStatuses.SERVER_ERROR),
	HANDLER_SERVICE_REQUIRED(50001004, "Failed to register handler method '{}': No service name specified.", ErrorStatuses.SERVER_ERROR),
	HANDLER_EVENT_REQUIRED(50001005, "Failed to register handler method '{}': No event definition in annotations or arguments.", ErrorStatuses.SERVER_ERROR),
	EVENT_CONTEXT_EVENT_MISMATCH(50001006, "EventContext '{}' is bound to event '{}' and does not match event '{}'", ErrorStatuses.SERVER_ERROR),
	EVENT_CONTEXT_MISSING_ANNOTATION(50001007, "EventContext '{}' is missing required @EventName annotation", ErrorStatuses.SERVER_ERROR),
	EVENT_CONTEXT_ARGUMENT_MISSING_ANNOTATION(50001008, "Method '{}' has parameter of EventContext type '{}' which misses required @EventName annotation", ErrorStatuses.SERVER_ERROR),
	EVENT_CONTEXT_ARGUMENT_MULTIPLE_EVENTS(50001009, "Method '{}' has parameter of EventContext type '{}' but is registered for multiple events", ErrorStatuses.SERVER_ERROR),
	EVENT_CONTEXT_ARGUMENT_MISMATCH(50001010, "Method '{}' has parameter of EventContext type '{}' which does not exactly fit registered event '{}'", ErrorStatuses.SERVER_ERROR),
	EVENT_CONTEXT_ARGUMENT_DUPLICATES(50001011, "Method '{}' declares multiple EventContext arguments, but only a maximum of one is supported", ErrorStatuses.SERVER_ERROR),
	POJO_ARGUMENT_MISMATCH(50001012, "Event '{}' not supported, when requiring argument '{}' in method '{}'", ErrorStatuses.SERVER_ERROR),
	POJO_ARGUMENT_DUPLICATES(50001013, "Method '{}' declares multiple entity-based arguments, but only a maximum of one is supported", ErrorStatuses.SERVER_ERROR),
	RETURN_TYPE_MISMATCH(50001014, "Event '{}' not supported, when returning '{}' in method '{}'", ErrorStatuses.SERVER_ERROR),
	RESOLVING_PARAMETER_TYPE_FAILED(50001015, "Can't resolve parameter type '{}' of method '{}'", ErrorStatuses.SERVER_ERROR),
	RESOLVING_RETURN_TYPE_FAILED(50001016, "Can't resolve return type '{}' of method '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_TYPE(50001017, "'{}' is not a parameterized type", ErrorStatuses.SERVER_ERROR),

	// 500 (02: Events & Services)
	UNEXPECTED_EVENT(50002001, "Unexpected event '{}'", ErrorStatuses.SERVER_ERROR),
	UNKNOWN_EVENT_PHASE(50002002, "Unknown event phase '{}'", ErrorStatuses.SERVER_ERROR),
	EVENT_NAME_REQUIRED(50002003, "Event name must not be null or empty", ErrorStatuses.SERVER_ERROR),
	SERVICE_NAME_REQUIRED(50002004, "Service name must not be null or empty", ErrorStatuses.SERVER_ERROR),
	DUPLICATE_SERVICE(50002005, "Can't register service with duplicate service name '{}'", ErrorStatuses.SERVER_ERROR),

	// 500 (03: ChangeSetContext & RequestContext)
	CHANGESET_CANCELATION_FAILED(50003001, "Unexpected exception during cancelation of member '{}' in ChangeSet '{}'", ErrorStatuses.SERVER_ERROR),
	CHANGESET_COMPLETION_FAILED(50003002, "Unexpected exception during completion of member '{}' in ChangeSet '{}'", ErrorStatuses.SERVER_ERROR),
	MULTIPLE_CHANGE_SET_MEMBERS(50003003, "A member is already registered on ChangeSet '{}': Only a single member is supported", ErrorStatuses.SERVER_ERROR),
	PLAIN_REQUEST_CONTEXT_NESTING(50003004, "Can't nest PlainRequestContextHolder", ErrorStatuses.SERVER_ERROR),
	TRANSACTION_TENANT_MISMATCH(50003005, "Database transaction was started for tenant '{}', but now found active tenant '{}': Create dedicated ChangeSet Contexts for each tenant", ErrorStatuses.SERVER_ERROR),
	TEMPORARY_REQUEST_CONTEXT_MESSAGES(50003006, "A messages object cannot be retrieved from a temporary RequestContext", ErrorStatuses.SERVER_ERROR),
	FEATURETOGGLESINFO_OVERRIDE(50003008, "Setting FeatureTogglesInfo in a nested RequestContext is not allowed", ErrorStatuses.SERVER_ERROR),
	TRANSACTION_INITIALIZATION_FAILED(50003009, "Failed to start a new transaction", ErrorStatuses.SERVER_ERROR),
	DUPLICATE_CHANGE_SET_MEMBERS(50003010, "A member with name '{}' is already registered on ChangeSet '{}'", ErrorStatuses.SERVER_ERROR),

	// 500 (04: Invalid model-related configuration)
	INVALID_WHERE_CONDITION(50004001, "Invalid where condition '{}' for entity '{}', event '{}' and user '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_CSV_FILE(50004002, "Failed to read CSV file from path '{}'", ErrorStatuses.SERVER_ERROR),
	NO_PARENT_ENTITY(50004003, "Failed to find parent entity of '{}'", ErrorStatuses.SERVER_ERROR),
	NO_KEYS_IN_RESULT(50004004, "Failed to determine keys of entity '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_ANNOTATION(50004005, "Invalid annotation '{}' for element '{}' in entity '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_CSN(50004006, "Failed to load CDS model from CSN file at resource path '{}'", ErrorStatuses.SERVER_ERROR),
	INCONSISTENT_WHERE_CONDITION(50004007, "No CXN expression found for where condition '{}' used for instance-based authorization of entity '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_BATCH_UPDATE(50004008, "Invalid batched update: Parameter value list size {} does not match batch size {}", ErrorStatuses.SERVER_ERROR),

	// 500 (05: Downstream services)
	NO_SERVICE_BINDING(50005001, "No service binding with tag '{}' found", ErrorStatuses.SERVER_ERROR),
	MULTIPLE_SERVICE_BINDING(50005002, "Multiple service bindings with tag '{}' found: Only a single one is allowed", ErrorStatuses.SERVER_ERROR),
	NO_DATA_SOURCE_PROVIDER(50005003, "Could not build SQL DataSource from service binding '{}'", ErrorStatuses.SERVER_ERROR),
	JDBC_CONNECTION_FAILED(50005004, "Can't obtain JDBC SQL connection", ErrorStatuses.SERVER_ERROR),
	INVALID_SERVICE_NAME(50005005, "Could not find service binding with name '{}'", ErrorStatuses.SERVER_ERROR),

	// 500 (06: Multi-Tenancy)
	SUBSCRIBER_FAILED(50006001, "Could not initialize multitenant subscription client", ErrorStatuses.SERVER_ERROR),
	INSTANCE_MANAGER_CLIENT_FAILED(50006002, "Could not create Instance Manager or Service Manager client from service binding '{}'", ErrorStatuses.SERVER_ERROR),
	SUBSCRIPTION_FAILED(50006003, "Failed to process subscription of tenant '{}'", ErrorStatuses.SERVER_ERROR),
	UNSUBSCRIPTION_FAILED(50006004, "Failed to cancel subscription of tenant '{}'", ErrorStatuses.SERVER_ERROR),
	DEPLOYMENT_FAILED(50006005, "Failed to deploy schema for tenant(s) '{}'", ErrorStatuses.SERVER_ERROR),
	JOB_STATUS_UPDATE_FAILED(50006006, "Failed to update status for asynchronous schema deployment job '{}'", ErrorStatuses.SERVER_ERROR),
	SUBSCRIBE_CALLBACK_FAILED(50006007, "Failed to call SaaS Provisioning service after subscribe request of tenant '{}' finished", ErrorStatuses.SERVER_ERROR),
	UNSUBSCRIBE_CALLBACK_FAILED(50006008, "Failed to call SaaS Provisioning service after unsubscribe request of tenant '{}' finished", ErrorStatuses.SERVER_ERROR),
	TENANT_READ_FAILED(50006010, "Failed to read tenant ids", ErrorStatuses.SERVER_ERROR),
	EDMX_READ_FAILED(50006011, "Failed to read EDMX for service '{}' from MTX sidecar", ErrorStatuses.SERVER_ERROR),
	MT_LIB_SETUP_FAILED(50006012, "Could not setup multi tenancy library: '{}'", ErrorStatuses.SERVER_ERROR),
	NO_UNIQUE_DATASOURCE_SERVICE(50006013, "Failed to determine a unique service binding to configure the primary datasource. Please specify cds.dataSource.binding", ErrorStatuses.SERVER_ERROR),
	NO_PROVISIONINGSERVICE_URL(50006014, "No url for provisioning service specified", ErrorStatuses.SERVER_ERROR),

	// 500 (07: Messaging)
	INVALID_CLOUDEVENTS_MESSAGE(50007002, "Failed to parse message as CloudEvents message", ErrorStatuses.SERVER_ERROR),
	INVALID_BROKER_CONFIGURATION(50007006, "Invalid messaging broker configuration for service '{}'", ErrorStatuses.SERVER_ERROR),
	EVENT_PROCESSING_FAILED(50007009, "Failed to process event '{}' on service '{}' received on queue '{}'", ErrorStatuses.SERVER_ERROR),
	EVENT_EMITTING_FAILED(50007011, "Failed to send message to topic '{}'", ErrorStatuses.SERVER_ERROR),
	NO_WEBHOOK_URL(50007012, "Failed to determine application URL for webhook registration", ErrorStatuses.SERVER_ERROR),
	NO_TENANT_INFO(50007013, "Required tenant information is missing in the request context", ErrorStatuses.SERVER_ERROR),
	UNKNOWN_TENANT(50007014, "Could not determine subdomain for tenant '{}'", ErrorStatuses.NOT_FOUND),
	TENANT_ADMIN_FORBIDDEN(50007015, "Not authorized for enterprise-messaging tenant management", ErrorStatuses.FORBIDDEN),
	UPDATE_TENANT_IS_RUNNING(50007016, "The tenant update is currently running", ErrorStatuses.CONFLICT),
	TENANT_SYNC_SUBSCRIBE_FORBIDDEN(50007017, "Only asynchronous SaaS subscription is supported when using enterprise-messaging with multitenancy", ErrorStatuses.SERVER_ERROR),
	MULTIPLE_MESSAGINGSERVICE(50007019, "Multiple messaging services found: Consider configuring the composite messaging service", ErrorStatuses.SERVER_ERROR),
	INVALID_DATA_FORMAT(50007020, "Invalid data structure of the remote event which is not matching the CDS event definition '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_SSL_CONTEXT(50007021, "Failed to create the mTLS context", ErrorStatuses.SERVER_ERROR),
	ACKNOWLEDGMENT_FAILED(50007022, "Failed to acknowledge message with topic '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_STRING_MAP_COMBINATION(50007023, "Either data string or the two maps data and headers may be set, not both", ErrorStatuses.SERVER_ERROR),
	NO_MESSAGE_PROVIDED(50007024, "Neither a data string nor the two maps data and headers are set", ErrorStatuses.SERVER_ERROR),
	INVALID_KAFKA_CLUSTER_PROVIDED(50007025, "The Kafka cluster could not be determined from the Kafka credentials", ErrorStatuses.SERVER_ERROR),

	// 500 (08: OData V4)
	INVALID_METADATA(50008001, "Could not load a valid OData V4 metadata file for service '{}'", ErrorStatuses.SERVER_ERROR),
	RESPONSE_SERIALIZATION_FAILED(50008002, "Failed to serialize response payload", ErrorStatuses.SERVER_ERROR),
	DESERIALIZER_FAILED(50008003, "Failed to create deserializer", ErrorStatuses.SERVER_ERROR),
	SERIALIZER_FAILED(50008004, "Failed to create serializer", ErrorStatuses.SERVER_ERROR),
	UNEXPECTED_URI_RESOURCE(50008005, "Unexpected URI resource type '{}'", ErrorStatuses.SERVER_ERROR),
	UNEXPECTED_EDM_TYPE(50008006, "Unexpected EDM type '{}'", ErrorStatuses.SERVER_ERROR),
	ORDERBY_PARSING_FAILED(50008007, "Failed to transform $orderby to CQN", ErrorStatuses.SERVER_ERROR),
	FILTER_PARSING_FAILED(50008008, "Failed to transform $filter to CQN", ErrorStatuses.SERVER_ERROR),
	SELECT_PARSING_FAILED(50008009, "Failed to transform $select to CQN", ErrorStatuses.SERVER_ERROR),
	MISSING_VALUE_FOR_COUNT(50008010, "The query result misses a value for $count", ErrorStatuses.SERVER_ERROR),
	VALUE_PARSING_FAILED(50008011, "Failed to transform expression to CQN value", ErrorStatuses.SERVER_ERROR),
	UNEXPECTED_PROPERTY_TYPE(50008013, "Unexpected EDM property type '{}' of property '{}'", ErrorStatuses.SERVER_ERROR),
	MISSING_BINDING_TARGET(50008014, "Missing EdmBindingTarget of EDM type '{}'", ErrorStatuses.SERVER_ERROR),
	WRONG_BASE_EDM_TYPE(50008015, "Wrong base EDM type '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_I18N_RESOURCES_FORMAT(50008016, "The EDMX I18N resources '{}' could not be loaded", ErrorStatuses.SERVER_ERROR),

	// 500 (09: OData V2)
	SEARCH_PARSING_FAILED(50009001, "Failed to transform custom 'search' option to CQN", ErrorStatuses.SERVER_ERROR),
	INVALID_METADATA_V2(50009002, "File '{}' is not a valid V2 metadata file", ErrorStatuses.SERVER_ERROR),
	RESOLVING_FUNCTION_IMPORT_ANNOTATION_FAILED(50009003, "Failed to resolve annotations for function import '{}'", ErrorStatuses.SERVER_ERROR),
	MISSING_EDM_PROPERTY(50009004, "Failed to get EDM property '{}' from EDM entity/type '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_METADATA_V2_DYNAMIC(50009005, "Dynamic model provider did not respond with a valid V2 metadata file for service '{}'", ErrorStatuses.SERVER_ERROR),
	VALUE_CONVERSION_FAILED(50009006, "Could not convert value '{}' of type '{}' to EDM type '{}'", ErrorStatuses.SERVER_ERROR),

	// 500 (10: CSV-import)
	INVALID_CSV_FILE_ENTITYNOTFOUND(50010001, "Can't find an entity '{}' matching CSV file '{}'.", ErrorStatuses.SERVER_ERROR),
	INVALID_CSV_FILE_INVALIDHEADER(50010003, "CSV file '{}' has header line with multiple delimiters", ErrorStatuses.SERVER_ERROR),
	INVALID_CSV_FILE_UNKNOWNCOLUMN(50010004, "CSV file '{}' has column '{}' which does not exist on entity '{}'", ErrorStatuses.SERVER_ERROR),
	INVALID_CSV_FILE_TYPEMISMATCH(50010005, "Value '{}' does not match type '{}' of column '{}' in CSV file '{}'", ErrorStatuses.SERVER_ERROR),

	// 500 (11: CDS Runtime Configurer)
	CONFIGURER_COMPLETED(50011001, "The CDS runtime configurer is already completed: Changes to the underlying CDS runtime are no longer allowed", ErrorStatuses.SERVER_ERROR),
	INVALID_CONFIGURATION_PHASE(50011002, "An operation of phase '{}' is not allowed anymore, as the CDS runtime configurer is already in phase '{}'", ErrorStatuses.SERVER_ERROR),

	// 500 (12: Remote OData)
	REMOTE_ODATA_PATH_EXPR(50012001, "Could not transform CQN to OData request: Accessing elements through path expressions is not supported by the OData protocol", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_REF_TYPE(50012003, "Could not transform CQN to OData request: Failed to determine the type of the reference '{}' in entity '{}'", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_TYPE_CONV(50012004, "Could not convert object '{}' of type '{}' to CDS base type '{}'", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_ERROR_RESPONSE(50012005, "The remote OData service responded with status code '{}'", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_FUNCTION(50012006, "Could not transform CQN to OData request: The function '{}' is not supported", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_NUM_ARGS(50012007, "Could not transform CQN to OData request: Wrong number of arguments was provided for function '{}'", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_OPERATOR(50012008, "Could not transform CQN to OData request: The operator '{}' is used incorrectly", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_FUNCTION_VERSION(50012009, "Could not transform CQN to OData request: The function '{}' is not supported for OData version '{}'", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_V2_LAMBDA(50012010, "Could not transform CQN to OData request: Lambda operators are not supported in OData V2", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_JSON_SERIALIZATION(50012011, "Could not serialize entry to OData compatible JSON", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_ONLY_COLLECTION(50012012, "Could not transform CQN to OData request: POST requests are only supported on collections", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_ONLY_ENTITY(50012013, "Could not transform CQN to OData request: PATCH/DELETE requests are only supported on single entities", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_INCOMPLETE_KEY(50012014, "Could not transform CQN to OData request: Could not retrieve values for all keys of entity '{}'", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_INVALID_LAMBDA_ALL(50012015, "Could not transform CQN to OData request: Lambda 'all' statements without argument are not supported", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_MISSING_STATEMENT(50012016, "Could not transform CQN to OData request: CQN Select Statement defining the target of the operation is missing", ErrorStatuses.SERVER_ERROR),
	REMOTE_ODATA_EMPTY_SELECTION(50012017, "Could not transform CQN to OData request: Empty select list. All elements have been excluded.", ErrorStatuses.SERVER_ERROR),


	// 500 (13: Composite)
	NO_SERVICE_IN_CATALOG(50013001, "Could not find destination service with name '{}', configured for composite service '{}', in the service catalog", ErrorStatuses.SERVER_ERROR),
	NO_DESTINATION_SERVICE(50013002, "No destination service defined for the event '{}' in the composite service configuration '{}'", ErrorStatuses.SERVER_ERROR),

	// 500 (14: Kubernetes)
	INVALID_SECRETS_PATH(50014001, "Secrets path '{}' of service binding '{}' is not a directory", ErrorStatuses.SERVER_ERROR),
	INVALID_PROPERTY_FILE(50014002, "Could not read property file '{}'", ErrorStatuses.SERVER_ERROR),
	APP_PROPERTIES_UNSUPPORTED(50014003, "Kubernetes does not provide an application name or ID", ErrorStatuses.SERVER_ERROR),

	// 500 (15: Cloud Foundry)
	MULTIPLE_APPLICATION_URLS(50015001, "Cannot determine the application url because multiple URIs are provided by VCAP_APPLICATION", ErrorStatuses.SERVER_ERROR),

	// 500 (16: Audit Log Service)
	AUDITLOG_SERVICE_NOT_AVAILABLE(50016001, "Auditlog service is not reachable", ErrorStatuses.SERVER_ERROR),
	AUDITLOG_SERVICE_INVALID_MESSAGE(50016002, "Invalid auditlog message", ErrorStatuses.SERVER_ERROR),
	AUDITLOG_SERVICE_INVALID_CONFIG(50016003, "Auditlog service configuration is invalid, parameter '{}' is missing", ErrorStatuses.SERVER_ERROR),
	AUDITLOG_SERVICE_NO_USER(50016004, "User information is missing, can't log audit message", ErrorStatuses.SERVER_ERROR),
	AUDITLOG_SERVICE_INVALID_PLAN(50016005, "Auditlog service plan '{}' not supported", ErrorStatuses.SERVER_ERROR),
	NO_AUDITLOG_MODEL(50016006, "Auditlog model '{}' is not available", ErrorStatuses.SERVER_ERROR),
	AUDITLOG_NOT_READABLE(50016007, "Auditlog model '{}' is not readable", ErrorStatuses.SERVER_ERROR),
	AUDITLOG_DATA_SUBJECT_MISSING(50016008, "A data subject for entity '{}' is missing, can't log audit message", ErrorStatuses.SERVER_ERROR),
	AUDITLOG_UNEXPECTED_HTTP_STATUS(50016010, "Audit Log service returned with unexpected status code {}", ErrorStatuses.SERVER_ERROR),
	// 500 (17: OutBox)
	CONTEXT_SERIALIZATION_FAILED(50017001, "Failed to serialize the event context of event '{}'", ErrorStatuses.SERVER_ERROR),
	CONTEXT_DESERIALIZATION_FAILED(50017002, "Failed to deserialize the outbox message of target event '{}'", ErrorStatuses.SERVER_ERROR),

	// 500 (18: Remote HCQL)
	REMOTE_HCQL_ERROR_RESPONSE(50018001, "Remote HCQL service responded with HTTP status code '{}', error code '{}' and message '{}'", ErrorStatuses.SERVER_ERROR),
	REMOTE_HCQL_COMMUNICATION_FAILED(50018002, "Remote HCQL communication failed", ErrorStatuses.SERVER_ERROR),

	// 500 (19: PDF Service)
	PDF_SERVICE_NOT_AVAILABLE(50019001, "No PDF export service available", ErrorStatuses.SERVER_ERROR),
	ERROR_GENERATING_PDF(50019002, "Failed to generate PDF", ErrorStatuses.SERVER_ERROR),
	NO_PDF_DESC_HEADER(50019003, "No PDF descriptor provided to generate a document", ErrorStatuses.SERVER_ERROR),
	INVALID_PDF_DESC_HEADER(50019004, "Invalid PDF descriptor header", ErrorStatuses.SERVER_ERROR),

	// 501
	UNSUPPORTED_ACTION_FUNCTION_RETURN_TYPE(501001, "Action or function return values of type '{}' are not yet supported", ErrorStatuses.NOT_IMPLEMENTED),
	UNSUPPORTED_METHOD(501002, "Method '{}' is not yet supported", ErrorStatuses.NOT_IMPLEMENTED),
	UNSUPPORTED_OPERATOR(501003, "Operator '{}' is not yet supported", ErrorStatuses.NOT_IMPLEMENTED),
	UNSUPPORTED_EXPAND_ALL_IN_DRAFT(501004, "Expanding all fields on a draft enabled entity is not yet supported", ErrorStatuses.NOT_IMPLEMENTED),
	UNSUPPORTED_PATH_IN_DRAFT(501005, "Expanding or inlining a draft enabled entity via a path is not supported", ErrorStatuses.NOT_IMPLEMENTED),
	UNSUPPORTED_APPLY_OPTION(501006, "Apply option '{}' is not yet supported", ErrorStatuses.NOT_IMPLEMENTED),
	UNSUPPORTED_CONCAT_LAST(501007, "'concat' is currently only supported as the last transformation of $apply", ErrorStatuses.NOT_IMPLEMENTED),
	UNSUPPORTED_CONCAT_SYSTEMQUERY(501008, "Transformation 'concat' with system query option {} is not supported", ErrorStatuses.NOT_IMPLEMENTED),
	UNSUPPORTED_RESTRICTION(501009, "Restriction on CDS model element '{}' of type {} is not supported", ErrorStatuses.NOT_IMPLEMENTED);

	private final int code;
	private final String description;
	private final ErrorStatus httpError;

	private CdsErrorStatuses(int code, String description, ErrorStatus httpError) {
		this.code = code;
		this.description = description;
		this.httpError = httpError;
	}

	@Override
	public String getCodeString() {
		return Integer.toString(code);
	}

	@Override
	public String getDescription() {
		return description;
	}

	@Override
	public int getHttpStatus() {
		return httpError.getHttpStatus();
	}

	public ErrorStatus getHttpError() {
		return httpError;
	}

}
