/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.CdsCommunicationException;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.model.CdsAnnotations;

/**
 * Utility class for the draft handling.
 *
 */
public class DraftUtils {

	private static final Logger log = LoggerFactory.getLogger(DraftUtils.class);

	private DraftUtils() {
	}

	/**
	 * Checks if the service has a draft enabled entity.
	 * @param serviceName the service name
	 * @param model the model
	 * @return {@code true} if the service has a draft enabled entity
	 */
	public static boolean isDraftEnabled(String serviceName, CdsModel model) {
		return model.findService(serviceName).map(s -> s.entities().anyMatch(e -> isDraftEnabled(e))).orElse(false);
	}

	/**
	 * Checks is the ref points to a draft instance.
	 * @param ref the entity ref
	 * @param entity the entity type
	 * @param model the model
	 * @return true, if the ref points to a draft instance
	 */
	public static boolean isDraftTarget(CqnStructuredTypeRef ref, CdsEntity entity, CdsModel model) {
		if(DraftUtils.isDraftEnabled(entity)) {
			Iterator<ResolvedSegment> segments = CqnAnalyzer.create(model).analyze(ref).reverse();
			while(segments.hasNext()) {
				Map<String, Object> targetKeys = segments.next().keyValues();
				if(!targetKeys.isEmpty()) {
					return targetKeys.containsKey(Drafts.IS_ACTIVE_ENTITY) && Boolean.FALSE.equals(targetKeys.get(Drafts.IS_ACTIVE_ENTITY));
				}
			}
		}
		return false;
	}

	/**
	 * Checks if the entity is draft enabled.
	 * @param entity the entity
	 * @return {@code true} if the entity is draft enabled
	 */
	public static boolean isDraftEnabled(CdsAnnotatable entity) {
		return CdsAnnotations.DRAFT_ANNOTATION.isTrue(entity) ||
				CdsAnnotations.DRAFT_PREPARE_ANNOTATION.getOrDefault(entity) != null;
	}

	public static boolean isDraftRoot(CdsAnnotatable entity) {
		return CdsAnnotations.DRAFT_ANNOTATION.isTrue(entity);
	}

	public static void gcDraftsOfAllServices(CdsRuntime runtime) {
		Stream<DraftService> draftServices = runtime.getServiceCatalog().getServices(DraftService.class);
		draftServices.forEach(DraftService::gcDrafts);
	}

	public static void gcDraftsOfAllServicesAndTenants(CdsRuntime runtime) {
		TenantProviderService tenantProvider = runtime.getServiceCatalog().getService(TenantProviderService.class, TenantProviderService.DEFAULT_NAME);
		List<String> tenants = tenantProvider.readTenants();
		tenants.forEach(tenant -> {
			try {
				runtime.requestContext().systemUser(tenant).privilegedUser().run(context -> {
					gcDraftsOfAllServices(runtime);
				});
			} catch (Exception e) {
				Throwable rootCause = ExceptionUtils.getRootCause(e);
				if (rootCause instanceof CdsCommunicationException && ((CdsCommunicationException) rootCause).getHttpStatusCode() == 404) {
					log.debug("Skipped draft GC for nonexistent tenant '{}'", tenant);
				} else {
					log.error("Failed to gc drafts of tenant '{}'", tenant, e);
				}
			}
		});
	}

}
