/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/

package com.sap.cds.services.utils;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.environment.CdsProperties.Locales.Normalization;

/**
 * Utility class to handle the locale settings
 */
public class LocaleUtils {

	private final static Logger logger = LoggerFactory.getLogger(LocaleUtils.class);

	/**
	 * Prefix for the localized entities and views
	 */
	private final static String LOCALIZED_PREFIX = "localized";

	/**
	 * SAP-specific locales
	 */
	private final static String SAPTRC_ALIAS = "1Q";
	private final static Locale SAPTRC_LOCALE = Locale.forLanguageTag("en-US-x-saptrc");
	private final static Locale SAPTRC_BUNDLE = Locale.forLanguageTag("en-US-saptrc");

	private final static String SAPPSD_ALIAS = "2Q";
	private final static Locale SAPPSD_LOCALE = Locale.forLanguageTag("en-US-x-sappsd");
	private final static Locale SAPPSD_BUNDLE = Locale.forLanguageTag("en-US-sappsd");

	private final static String SAPRIGI_ALIAS = "3Q";
	private final static Locale SAPRIGI_LOCALE = Locale.forLanguageTag("en-US-x-saprigi");
	private final static Locale SAPRIGI_BUNDLE = Locale.forLanguageTag("en-US-saprigi");

	private final static List<Locale> DEFAULT_INCLUDE_LIST = Arrays.asList(
		Locale.forLanguageTag("zh-CN"),
		Locale.forLanguageTag("zh-HK"),
		Locale.forLanguageTag("zh-TW"),
		Locale.forLanguageTag("en-GB"),
		Locale.forLanguageTag("fr-CA"),
		Locale.forLanguageTag("pt-PT"),
		Locale.forLanguageTag("es-CO"),
		Locale.forLanguageTag("es-MX"),
		SAPTRC_LOCALE,
		SAPPSD_LOCALE,
		SAPRIGI_LOCALE
		);

	/**
	 * The include list of language tags that will not be normalized. For all other tags, only
	 * the language will be considered
	 */
	private final Set<Locale> localeIncludeList = new HashSet<>();

	public LocaleUtils(CdsProperties properties) {
		// initially calculate locale include list
		calculateLocaleIncludeList(properties.getLocales().getNormalization());
	}

	// used only in tests -> package visibility intended
	void calculateLocaleIncludeList(Normalization config) {
		localeIncludeList.clear();

		// add default include list entries
		if (config.isDefaults()) {
			localeIncludeList.addAll(DEFAULT_INCLUDE_LIST);
		}
		// add custom include list entries
		config.getIncludeList().stream().map(t -> Locale.forLanguageTag(t)).forEach(localeIncludeList::add);
	}

	/**
	 * Determines the locale for a request.
	 *
	 * @param sapLocaleQueryParam Value of "sap-locale" query parameter
	 * @param acceptLanguageHeader Value of accepted language header parameter
	 * @param sapLanguageQueryParam	Value of "sap-language"query parameter
	 * @param xSapLanguageHeader Value of "x-sap-request-language" query parameter
	 *
	 * @return the locale
	 */
	public Locale getLocale(String sapLocaleQueryParam, String acceptLanguageHeader, String sapLanguageQueryParam, String xSapLanguageHeader) {

		// proprietary SAP URL parameter to set the language
		Locale locale = null;
		if (!StringUtils.isEmpty(sapLocaleQueryParam)) {
			locale = Locale.forLanguageTag(sapLocaleQueryParam);
		} else if (!StringUtils.isEmpty(acceptLanguageHeader)) {
			try {
				locale = Locale.LanguageRange.parse(acceptLanguageHeader).stream()
						.map(range -> Locale.forLanguageTag(range.getRange())).findFirst().orElse(null);
			} catch(IllegalArgumentException e) {
				logger.debug("Omitting invalid accept header '{}': {} ", acceptLanguageHeader, e.getMessage());
			}
		} else if (!StringUtils.isEmpty(sapLanguageQueryParam)) {
			// handle special SAP language codes
			if (sapLanguageQueryParam.equals(SAPTRC_ALIAS)) {
				locale = SAPTRC_LOCALE;
			} else if (sapLanguageQueryParam.equals(SAPPSD_ALIAS)){
				locale = SAPPSD_LOCALE;
			} else if (sapLanguageQueryParam.equals(SAPRIGI_ALIAS)){
				locale = SAPRIGI_LOCALE;
			} else {
				// if the localeStr is invalid this leads to an "empty" locale which behaves the same as no locale at all
				locale = Locale.forLanguageTag(sapLanguageQueryParam);
			}
		}
		// proprietary SAP header to set the language
		else if (!StringUtils.isEmpty(xSapLanguageHeader)) {
			locale = Locale.forLanguageTag(xSapLanguageHeader);
		}

		// normalize locales except they are contained in the include list
		if (locale != null && !localeIncludeList.contains(locale)) {
			locale = new Locale(locale.getLanguage());
		}
		return locale;
	}

	/**
	 * Calculates the localized entity name that can be used to access localized
	 * fields
	 * @param entity the entity name
	 * @return the localized entity name
	 */
	public static String getLocalizedEntityName(String entity) {
		return LOCALIZED_PREFIX + "." + entity;
	}

	/**
	 * Returns the correct locale String for the sap-language query parameter
	 * @param locale the locale
	 * @return the correct locale String for the sap-language query parameter
	 */
	public static String getLocaleStringForQuery(Locale locale) {
		if (SAPTRC_LOCALE.equals(locale)) {
			return SAPTRC_ALIAS;
		}
		if (SAPPSD_LOCALE.equals(locale)) {
			return SAPPSD_ALIAS;
		}
		if (SAPRIGI_LOCALE.equals(locale)) {
			return SAPRIGI_ALIAS;
		}
		return locale.toLanguageTag();
	}

	/**
	 * Determines the locale used for resource bundle lookups
	 * @param locale the locale
	 * @return the locale to beused for bundle lookup
	 */
	public static Locale getLocaleForBundle(Locale locale) {
		if(SAPTRC_LOCALE.equals(locale)) {
			return SAPTRC_BUNDLE;
		} else if(SAPPSD_LOCALE.equals(locale)) {
			return SAPPSD_BUNDLE;
		} else if(SAPRIGI_LOCALE.equals(locale)) {
			return SAPRIGI_BUNDLE;
		}
		return locale;
	}

}
