package com.sap.cds.services.utils;

import org.slf4j.MDC;

/**
 * Utility for retrieving/setting the tenant id from/into the MDC.
 */
public class TenantIdUtils {

	public static final String TENANT_ID_FIELD = "tenant_id";

	public static final String TENANT_ID_UNKNOWN = "-"; // used by cf-java-logging-support as "unknown" value

	private TenantIdUtils() {
	}

	/**
	 * Checks whether there's a tenant id entry in the {@link MDC}.
	 * 
	 * @return true, if there's an entry under the key "tenant_id"
	 */
	public static boolean mdcHasEntry() {
		return !isUnknown(getFromMDC());
	}

	/**
	 * Puts the provided tenant id into the {@link MDC} under the key
	 * "tenant_id".
	 * 
	 * @param tenantId the tenant id
	 */
	public static void putInMDC(String tenantId) {
		if (!isUnknown(tenantId)) {
			MDC.put(TENANT_ID_FIELD, tenantId);
		}
	}

	/**
	 * Gets a tenant id value from the MDC, if existing, or null otherwise.
	 * 
	 * @return a tenant id
	 */
	public static String getFromMDC() {
		return MDC.get(TENANT_ID_FIELD);
	}

	/**
	 * Removes the tenant id field from the {@link MDC}.
	 */
	public static void clearMDC() {
		MDC.remove(TENANT_ID_FIELD);
	}

	private static boolean isUnknown(String tenantId) {
		return StringUtils.isEmpty(tenantId) || TENANT_ID_UNKNOWN.equals(tenantId);
	}
}
