package com.sap.cds.services.utils.model;

import java.time.Duration;
import java.util.List;
import java.util.function.Function;

import org.apache.http.HttpHeaders;

import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.mtx.impl.Authenticator;
import com.sap.cds.mtx.impl.CacheParams;
import com.sap.cds.mtx.impl.ClientCredentialJwtAccess;
import com.sap.cds.mtx.impl.ClientCredentialJwtReader;
import com.sap.cds.mtx.impl.MetaDataAccessorImpl;
import com.sap.cds.mtx.impl.MetaDataAccessorImpl.EdmxModelCreator;
import com.sap.cds.mtx.impl.ModelProviderAccess;
import com.sap.cds.mtx.impl.SidecarAccess;
import com.sap.cds.mtx.impl.SidecarAccessV1;
import com.sap.cds.mtx.impl.XsuaaParams;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.environment.CdsProperties.Model.Provider;
import com.sap.cds.services.environment.CdsProperties.MultiTenancy.Sidecar;
import com.sap.cds.services.environment.CdsProperties.MultiTenancy.Sidecar.Cache;
import com.sap.cds.services.mt.ExtensibilityService;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.XsuaaUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.mt.tools.api.RequestEnhancer;
import com.sap.cloud.mt.tools.api.ResilienceConfig;

public class DynamicModelUtils {

	private final CdsRuntime runtime;
	private final Provider providerConfig;
	private final Sidecar sidecarConfig;
	private final XsuaaUtils xsuaaUtils;
	private final boolean mtxs;

	public DynamicModelUtils(CdsRuntime runtime) {
		this.runtime = runtime;
		this.providerConfig = runtime.getEnvironment().getCdsProperties().getModel().getProvider();
		this.sidecarConfig = runtime.getEnvironment().getCdsProperties().getMultiTenancy().getSidecar();
		this.mtxs = runtime.getEnvironment().getCdsProperties().getMultiTenancy().getMtxs().isEnabled();
		this.xsuaaUtils = new XsuaaUtils(runtime);
	}

	/**
	 * Returns true, if the static model, packaged with the application, can be loaded
	 * Retrieves the required information from the current {@link RequestContext}
	 *
	 * @return true, if the static model, packaged with the application, can be loaded
	 */
	public boolean useStaticModel() {
		RequestContext requestContext = RequestContext.getCurrent(runtime);
		return useStaticModel(requestContext.getUserInfo(), requestContext.getFeatureTogglesInfo());
	}

	/**
	 * Returns true, if the static model, packaged with the application, can be loaded
	 *
	 * @param userInfo the {@link UserInfo}
	 * @param featureTogglesInfo the {@link FeatureTogglesInfo}
	 * @return true, if the static model, packaged with the application, can be loaded
	 */
	public boolean useStaticModel(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
		boolean hasAllToggles = featureTogglesInfo.getEnabledFeatureToggles().anyMatch(t -> t.getName().trim().equals("*"));
		return userInfo.getTenant() == null && (!isModelProviderEnabled() || hasAllToggles);
	}

	/**
	 * @return {@code true} if ModelProviderService is configured
	 */
	public boolean isModelProviderEnabled() {
		return !StringUtils.isEmpty(providerConfig.getUrl()) ||
				(mtxs && !StringUtils.isEmpty(sidecarConfig.getUrl()));
	}

	public String getModelProviderUrl() {
		return !StringUtils.isEmpty(providerConfig.getUrl()) ? providerConfig.getUrl() : sidecarConfig.getUrl();
	}

	/**
	 * @return {@code true} if Classic MT Sidecar is configured
	 */
	public boolean isClassicSidecarEnabled() {
		return !mtxs &&	!StringUtils.isEmpty(sidecarConfig.getUrl());
	}

	/**
	 * Indicates whether the model is provided by a external service (i.e. it's dynamic) or not.
	 * Dynamic models can be served by:
	 * - classic sidecar
	 * - ModelProvider service (either as mtxs service or standalone)
	 *
	 * @return {@code true} if the model is dynamic
	 */
	public boolean isDynamicModelEnabled() {
		return isClassicSidecarEnabled() || isModelProviderEnabled();
	}

	public <T> MetaDataAccessor<T> createMetadataAccessor(EdmxModelCreator<T> strToEdmx, Function<String, CdsModel> strToModel) {
		// create accessor
		MetaDataAccessor<T> accessor = buildMetadataAccessor(strToEdmx, strToModel);

		// register accessor with extensibility service
		if(accessor != null) {
			ExtensibilityService extService = runtime.getServiceCatalog().getService(ExtensibilityService.class, ExtensibilityService.DEFAULT_NAME);
			if(extService != null) {
				extService.on(ExtensibilityService.EVENT_MODEL_CHANGED, null, context -> {
					String tenant = context.getUserInfo().getTenant();
					if (sidecarConfig.getCache().isRefreshEager()) {
						accessor.refresh(tenant);
					} else {
						accessor.evict(tenant);
					}
				});
			}
		}

		return accessor;
	}

	private <T> MetaDataAccessor<T> buildMetadataAccessor(EdmxModelCreator<T> strToEdmx, Function<String, CdsModel> strToModel) {
		RequestEnhancer authenticationEnhancer = getAuthenticationEnhancer();
		SidecarAccess sidecarAccess = null;
		CacheParams cacheParams = null;

		if (isModelProviderEnabled()) {
			sidecarAccess = new ModelProviderAccess(authenticationEnhancer, getResilienceConfig());
			cacheParams = getCacheParams(providerConfig.getCache());
		} else if (isClassicSidecarEnabled()) {
			cacheParams = getCacheParams(sidecarConfig.getCache());
			sidecarAccess = new SidecarAccessV1(authenticationEnhancer, getResilienceConfig());
		}

		if (sidecarAccess != null && cacheParams != null) {
			return new MetaDataAccessorImpl<>(new MetaDataAccessorImpl.MetaDataAccessorConfig.Builder()
					.sidecarAccess(sidecarAccess).cacheParams(cacheParams)
					.strToEdmx(strToEdmx).strToModel(strToModel)
					.build(), null);
		}
		return null;
	}

	private CacheParams getCacheParams(Cache cache) {
		long maximumSize = cache.getMaxSize();
		long expirationDuration = cache.getExpirationTime();
		long refreshDuration = cache.getRefreshTime();
		return new CacheParams(maximumSize, Duration.ofSeconds(expirationDuration), Duration.ofSeconds(refreshDuration), false);
	}

	public ResilienceConfig getResilienceConfig() {
		return ResilienceConfig.builder().numOfRetries(3).retryInterval(Duration.ofMillis(500)).build();
	}

	// TODO this should ideally be hidden behind the Cloud SDKs destinations
	public RequestEnhancer getAuthenticationEnhancer() {
		List<ServiceBinding> uaaBindings = xsuaaUtils.getXsuaaServiceBindings();
		if (!uaaBindings.isEmpty()) {
			ServiceBinding uaaInstance = uaaBindings.get(0);
			Authenticator authenticator = new ClientCredentialJwtAccess(new ClientCredentialJwtReader(new XsuaaParams(uaaInstance.getCredentials())));
			return request -> request.addHeader(HttpHeaders.AUTHORIZATION, authenticator.getAuthorization().get());
		}
		// should only happen during testing
		return request -> {};
	}

}
