package com.sap.cds.services.utils.outbox;

import java.time.Instant;
import java.util.Arrays;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.ServiceDelegator;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;

/**
 * Abstract implementation of {@link OutboxService}, providing base implementation for all specific outbox services.
 */
public abstract class AbstractOutboxService extends ServiceDelegator implements OutboxService {

	private final Set<String> eventsWithOnHandler = ConcurrentHashMap.newKeySet();

	public AbstractOutboxService(String name) {
		super(name);
	}

	@Override
	public void enroll(String targetEvent, String message) {
		OutboxMessageEventContext context = OutboxMessageEventContext.create(targetEvent);
		context.setMessage(message);
		emit(context);
	}

	@On
	@HandlerOrder(OrderConstants.On.PRIORITY)
	private void handleOutboxEvent(OutboxMessageEventContext context) {
		if (Boolean.FALSE.equals(context.getIsInbound())) {

			if (context.getTimestamp() == null) {
				context.setTimestamp(Instant.now());
			}

			enroll(context);
			context.setCompleted();
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.AUTO_COMPLETE)
	private void autoComplete(EventContext context) {
		// complete only for known target event registrations
		if (eventsWithOnHandler.contains(context.getEvent())) {
			context.setCompleted();
		}
	}

	@Override
	public void on(String[] events, String[] entities, int order, Handler handler) {
		super.on(events, entities, order, handler);
		Arrays.stream(events)
		.filter(event -> !StringUtils.isEmpty(event) && !event.equals("*"))
		.forEach(eventsWithOnHandler::add);
	}

	/**
	 * Stores the outbox message in the outbox, to be retrieved for later publishing.
	 * @param context the {@link OutboxMessageEventContext}
	 */
	protected abstract void enroll(OutboxMessageEventContext context);

}
