package com.sap.cds.services.utils.outbox;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;

public class OutboxUtils {

	public static final String OUTBOX_MODEL = "cds.outbox.Messages";

	/**
	 * Determines whether the outbox model is available.
	 *
	 * @param model CDS model
	 * @return <code>true</code> if outbox model is available and <code>false</code> otherwise.
	 */
	public static boolean hasOutboxModel(CdsModel model) {
		return model.findEntity(OUTBOX_MODEL).isPresent();
	}

	/**
	 * Determines the available outbox. Persistent outbox is preferred but if not available the in-memory is returned.
	 *
	 * @param runtime the {@link CdsRuntime}
	 * @return available outbox
	 */
	public static OutboxService chooseOutboxService(CdsRuntime runtime, boolean allowPersistentOutbox) {
		RequestContext context = RequestContext.getCurrent(runtime);
		OutboxService inMemoryOutbox = context.getServiceCatalog().getService(OutboxService.class, OutboxService.INMEMORY_NAME);
		OutboxService persistentOutbox = context.getServiceCatalog().getService(OutboxService.class, OutboxService.PERSISTENT_NAME);
		boolean providerTenantPersistence = runtime.getEnvironment().getCdsProperties().getOutbox().getPersistent().getProviderTenant().isEnabled();

		OutboxService result = allowPersistentOutbox ? persistentOutbox : null;
		// events of the provider tenant can't be stored, if MT is enabled, as there is no persistence for the provider.
		if (result == null || (!providerTenantPersistence && context.getUserInfo().getTenant() == null) || !hasOutboxModel(context.getModel())) {
			result = inMemoryOutbox;
		}
		return result;
	}
}
