/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.path;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Stream;

import com.sap.cds.reflect.CdsService;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.environment.CdsProperties.Application.ApplicationServiceConfig;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.model.CdsModelUtils;

/**
 * Provides a stream of {@link CdsResourcePath}s which are exposed by the {@link ApplicationService}s of a {@link ServiceCatalog}
 */
public class CdsServicePath {

	/**
	 * Creates a {@code Stream} of {@code CdsResourcePath} instances containing the paths of all exposed {@link ApplicationService}s given in the {@link ServiceCatalog} and configuration.
	 * Path in the configuration or annotations in the CDS service model are regarded with priority and override default paths reflecting the full qualified name of the service.
	 *
	 * @param runtime			The {@link CdsRuntime}, containing the {@link ServiceCatalog} with all {@link ApplicationService}s
	 * @param protocol			The protocol adapter for which these paths should be determined
	 *
	 * @return	A {@code Stream} of all {@code CdsResourcePath} instances of the given protocol adapter
	 */
	public static Stream<CdsResourcePath> servicePaths(CdsRuntime runtime, String protocol) {
		return calculateServicePaths(runtime, protocol).values().stream().flatMap(List::stream);
	}

	/**
	 * Creates a map of the base paths of each exposed service to their respective {@link ApplicationService} instance.
	 * The base paths are calculated for the given protocol adapter.
	 *
	 * @param runtime			The {@link CdsRuntime}, containing the {@link ServiceCatalog} with all {@link ApplicationService}s
	 * @param protocol			The protocol adapter for which these paths should be determined
	 * @return 	A map of base paths to their {@link ApplicationService}
	 */
	public static Map<String, ApplicationService> basePaths(CdsRuntime runtime, String protocol) {
		Map<String, ApplicationService> basePathMap = new HashMap<>();
		Map<ApplicationService, List<CdsResourcePath>> servicePaths = calculateServicePaths(runtime, protocol);
		for(Entry<ApplicationService, List<CdsResourcePath>> entry : servicePaths.entrySet()) {
			for(CdsResourcePath basePath : entry.getValue()) {
				basePathMap.put(basePath.getPath(), entry.getKey());
			}
		}
		return basePathMap;
	}

	private static Map<ApplicationService, List<CdsResourcePath>> calculateServicePaths(CdsRuntime runtime, String protocol) {
		Map<ApplicationService, List<CdsResourcePath>> servicePathMap = new HashMap<>();
		runtime.getServiceCatalog().getServices(ApplicationService.class).forEach(applicationService -> {
			CdsService cdsService = applicationService.getDefinition();
			ApplicationServiceConfig config = runtime.getEnvironment().getCdsProperties().getApplication().getService(applicationService.getName());
			List<String> endpointPaths = getEndpointPaths(protocol, cdsService, config);

			if(endpointPaths.isEmpty()) {
				return; // no endpoints exposed
			}

			CdsModelUtils modelUtils = new CdsModelUtils(runtime);
			boolean isPublicService = modelUtils.isPublic(cdsService).orElse(false);

			// calculate entity paths of the service (path will be relative to service path)
			List<CdsResourcePath> serviceSubPaths = new ArrayList<>();

			cdsService.entities().forEach(cdsEntity -> {
				boolean isPublicEntity = modelUtils.isPublic(cdsEntity).orElse(isPublicService);

				// add bound actions and functions
				List<CdsResourcePath> entitySubPaths = new ArrayList<>();
				cdsEntity.actions().forEach(cdsAction -> {
					CdsResourcePath actionPath = CdsResourcePathBuilder.cds(cdsAction).isPublic( modelUtils.isPublic(cdsAction).orElse(isPublicEntity) ).build();
					entitySubPaths.add(actionPath);
				});
				cdsEntity.functions().forEach(cdsFunction -> {
					CdsResourcePath actionPath = CdsResourcePathBuilder.cds(cdsFunction).isPublic( modelUtils.isPublic(cdsFunction).orElse(isPublicEntity) ).build();
					entitySubPaths.add(actionPath);
				});

				CdsResourcePath entityPath = null;
				if (isPublicEntity) {
					entityPath = CdsResourcePathBuilder.cds(cdsEntity).subPaths(entitySubPaths.stream()).isPublic(true).build();
				} else {
					List<String> publicEvents = modelUtils.getPublicEvents(cdsEntity);
					entityPath = CdsResourcePathBuilder.cds(cdsEntity).subPaths(entitySubPaths.stream()).isPublic(false).publicEvents(publicEvents.stream()).build();
				}
				serviceSubPaths.add(entityPath);
			});

			// add unbound actions and functions
			cdsService.actions().forEach(cdsAction -> {
				CdsResourcePath actionPath = CdsResourcePathBuilder.cds(cdsAction).isPublic( modelUtils.isPublic(cdsAction).orElse(isPublicService) ).build();
				serviceSubPaths.add(actionPath);
			});
			cdsService.functions().forEach(cdsFunction -> {
				CdsResourcePath functionPath = CdsResourcePathBuilder.cds(cdsFunction).isPublic( modelUtils.isPublic(cdsFunction).orElse(isPublicService) ).build();
				serviceSubPaths.add(functionPath);
			});

			// calculate the service paths
			List<CdsResourcePath> servicePaths = new ArrayList<>();
			for (String path : endpointPaths) {
				servicePaths.add(CdsResourcePathBuilder
					.cds(cdsService)
					.path(StringUtils.trim(path, '/'))
					.subPaths(serviceSubPaths.stream())
					.isPublic(isPublicService)
					.build());
			}
			
			servicePathMap.put(applicationService, servicePaths);
		});
		return servicePathMap;
	}

	private static List<String> getEndpointPaths(String protocol, CdsService cdsService, ApplicationServiceConfig config) {
		boolean serviceIgnoreAnnotation = CdsAnnotations.IGNORE.isTrue(cdsService);
		boolean serveIgnoreAnnotation = CdsAnnotations.SERVE_IGNORE.isTrue(cdsService);
		boolean serveIgnoreConfig = config.getServe().isIgnore();
		if (serviceIgnoreAnnotation || serveIgnoreAnnotation || serveIgnoreConfig) {
			// ignored services don't expose any endpoints
			return Collections.emptyList();
		}

		List<String> endpointPaths = new ArrayList<>();
		// endpoints from configuration
		config.getServe().getEndpoints()
			.stream()
			.filter(e -> protocol.equals(e.getProtocol()))
			.map(e -> e.getPath())
			.filter(p -> !StringUtils.isEmpty(p))
			.forEach(endpointPaths::add);

		if(!config.getServe().getEndpoints().isEmpty()) {
			// either return the configured endpoints for this protocol
			// or the still empty list (no endpoints for this protocol)
			return endpointPaths;
		}

		// endpoints from annotation
		List<Map<String, Object>> endpointAnnotation = CdsAnnotations.ENDPOINTS.getListOrDefault(cdsService);
		if(endpointAnnotation != null) {
			for(Map<String, Object> endpoint : endpointAnnotation) {
				if(protocol.equals(endpoint.get("protocol"))) {
					Object path = endpoint.get("path");
					if(path instanceof String && !StringUtils.isEmpty((String) path)) {
						endpointPaths.add((String) path);
					}
				}
			}

			if(!endpointAnnotation.isEmpty()) {
				// either return the configured endpoints for this protocol
				// or the still empty list (no endpoints for this protocol)
				return endpointPaths;
			}
		}


		// protocol defined in configuration
		List<String> protocolConfig = config.getServe().getProtocols();
		if(!protocolConfig.isEmpty() && !protocolConfig.stream().anyMatch(protocol::equals)) {
			// the service is explicitly not exposed via this protocol
			return Collections.emptyList();

		} else if (protocolConfig.isEmpty()) {
			// protocol defined in annotation
			List<String> protocolAnnotation = CdsAnnotations.PROTOCOLS.getListOrDefault(cdsService);
			if(protocolAnnotation != null && !protocolAnnotation.isEmpty() && !protocolAnnotation.stream().anyMatch(protocol::equals)) {
				// the service is explicitly not exposed via this protocol
				return Collections.emptyList();
			}
		}


		// path defined in configuration
		if(!StringUtils.isEmpty(config.getServe().getPath())) {
			endpointPaths.add(config.getServe().getPath());
		}

		if(!endpointPaths.isEmpty()) {
			return endpointPaths;
		}

		// paths defined in annotation
		List<String> pathAnnotation = CdsAnnotations.PATH.getListOrDefault(cdsService);
		if(pathAnnotation != null) {
			pathAnnotation.stream()
				.filter(p -> !StringUtils.isEmpty(p))
				.forEach(endpointPaths::add);
		}

		if(!endpointPaths.isEmpty()) {
			return endpointPaths;
		}

		// default is the qualified service name
		return Arrays.asList(config.getName());
	}

}
