/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.runtime;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;

import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CorrelationIdUtils;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.services.utils.LocaleUtils;
import com.sap.cds.services.utils.QueryParameters;
import com.sap.cds.services.utils.TemporalRangeUtils;

import jakarta.servlet.http.HttpServletRequest;

/**
 * {@link ParameterInfo} instance derived from current HTTP request.
 */
public class HttpParameterInfo implements ParameterInfo {

	private final Map<String, String[]> parameters;
	private final Map<String, String> headers;
	private final String correlationId;
	private final Locale locale;
	private final Instant validFrom;
	private final Instant validTo;

	private HttpParameterInfo(HttpServletRequest httpRequest, LocaleUtils localeUtils) {
		this.parameters = new HashMap<>(httpRequest.getParameterMap());
		this.headers = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
		Enumeration<String> headerKeys = httpRequest.getHeaderNames();
		if (headerKeys != null) {
			while (headerKeys.hasMoreElements()) {
				String headerKey = headerKeys.nextElement();
				Enumeration<String> headerValues = httpRequest.getHeaders(headerKey);
				String headerValue = null;
				if (headerValues != null && headerValues.hasMoreElements()) {
					headerValue = Collections.list(headerValues).stream().collect(Collectors.joining(","));
				}
				this.headers.put(headerKey, headerValue);
			}
		}

		this.correlationId = CorrelationIdUtils.getOrGenerateCorrelationId(this);
		this.locale = localeUtils.getLocale(
				getQueryParameter(QueryParameters.SAP_LOCALE),
				getHeader(HttpHeaders.ACCEPT_LANGUAGE),
				getQueryParameter(QueryParameters.SAP_LANGUAGE),
				getHeader(HttpHeaders.X_SAP_REQUEST_LANGUAGE_HEADER));

		Instant[] ranges = TemporalRangeUtils.getTemporalRanges(
				getQueryParameter(QueryParameters.VALID_FROM),
				getQueryParameter(QueryParameters.VALID_TO),
				getQueryParameter(QueryParameters.VALID_AT));
		this.validFrom = ranges[0];
		this.validTo = ranges[1];
	}

	/**
	 * Creates the {@link ParameterInfo} instance based on the current servlet (http) request.
	 * @param httpRequest	The http request
	 * @param runtime		The {@link CdsRuntime}
	 *
	 * @return	The {@link ParameterInfo} instance or {@code null} if there is no http request scope
	 */
	public static ParameterInfo fromRequest(HttpServletRequest httpRequest, CdsRuntime runtime) {
		return new HttpParameterInfo(httpRequest, new LocaleUtils(runtime.getEnvironment().getCdsProperties()));
	}

	@Override
	public String getCorrelationId() {
		return correlationId;
	}

	@Override
	public Map<String, String> getHeaders() {
		return Collections.unmodifiableMap(headers);
	}

	@Override
	public String getHeader(String id) {
		return headers.get(id);
	}

	@Override
	public String getQueryParameter(String key) {
		String[] values = parameters.get(key);
		if (values != null && values.length >= 1) {
			return values[0];
		}
		return null;
	}

	@Override
	public Map<String, String> getQueryParams() {
		Map<String, String> result = new HashMap<>();
		for (Map.Entry<String, String[]> entry : parameters.entrySet()) {
			String[] values = entry.getValue();
			String valueString = values != null ? Arrays.asList(values).stream().collect(Collectors.joining(",")) : null;
			result.put(entry.getKey(), valueString);
		}
		return result;
	}

	@Override
	public Locale getLocale() {
		return locale;
	}

	@Override
	public Instant getValidFrom() {
		return this.validFrom;
	}

	@Override
	public Instant getValidTo() {
		return this.validTo;
	}

}

