/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.services;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;

import java.util.List;
import java.util.Map;

import com.sap.cds.Result;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.ServiceDelegator;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CdsUpsertEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.model.CdsModelUtils;
import com.sap.cds.services.utils.model.CqnUtils;

public abstract class AbstractCqnService extends ServiceDelegator implements CqnService {

	protected final CdsRuntime runtime;

	protected AbstractCqnService(String name, CdsRuntime runtime) {
		super(name);
		this.runtime = runtime;
	}

	@Override
	public Result run(CqnSelect select, Object... paramValues) {
		return run(select, CqnUtils.convertToIndexMap(paramValues));
	}

	@Override
	public Result run(CqnSelect select, Map<String, Object> namedValues) {
		CdsReadEventContext context = CdsReadEventContext.create(getTargetEntity(select));

		context.setCqn(select);
		context.setCqnNamedValues(namedValues);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result run(CqnInsert insert) {
		CdsCreateEventContext context = CdsCreateEventContext.create(getTargetEntity(insert));

		context.setCqn(insert);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result run(CqnUpsert upsert) {
		CdsUpsertEventContext context = CdsUpsertEventContext.create(getTargetEntity(upsert));

		context.setCqn(upsert);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result run(CqnUpdate update, Object... paramValues) {
		return run(update, CqnUtils.convertToIndexMap(paramValues));
	}

	@Override
	public Result run(CqnUpdate update, Map<String, Object> namedValues) {
		return run(update, list(namedValues));
	}

	@Override
	public Result run(CqnUpdate update, Iterable<Map<String, Object>> valueSets) {
		CdsUpdateEventContext context = CdsUpdateEventContext.create(getTargetEntity(update));

		context.setCqn(update);
		context.setCqnValueSets(valueSets);

		emit(context);

		return context.getResult();
	}

	@Override
	public Result run(CqnDelete delete, Object... paramValues) {
		return run(delete, CqnUtils.convertToIndexMap(paramValues));
	}

	@Override
	public Result run(CqnDelete delete, Map<String, Object> namedValues) {
		return run(delete, list(namedValues));
	}

	@Override
	public Result run(CqnDelete delete, Iterable<Map<String, Object>> valueSets) {
		CdsDeleteEventContext context = CdsDeleteEventContext.create(getTargetEntity(delete));

		context.setCqn(delete);
		context.setCqnValueSets(valueSets);

		emit(context);

		return context.getResult();
	}

	protected List<Map<String, Object>> list(Map<String, Object> map) {
		return map.isEmpty() ? emptyList() : singletonList(map);
	}

	protected String getTargetEntity(CqnStatement statement) {
		if(statement.isSelect() && statement.asSelect().from().isSelect()) {
			return getTargetEntity(statement.asSelect().from().asSelect());
		}
		CdsModel model = RequestContext.getCurrent(runtime).getModel();
		return CdsModelUtils.getEntityPath(statement.ref(), model).target().type().getQualifiedName();
	}
}
