/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.environment.ServiceBindingUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;

public class IdentityUtils {

	private static final String XSUAA = "xsuaa";

	private static final String IDENTITY = "identity";

	private final CdsRuntime runtime;

	public IdentityUtils(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	/**
	 * @return	All found XSUAA bindings. In case a specific service binding is chosen via cds.security.xsuaa.binding,
	 * 			the list contains only this binding if available.
	 */
	public List<ServiceBinding> getXsuaaServiceBindings() {
		Stream<ServiceBinding> xsuaaBindings = runtime.getEnvironment().getServiceBindings().filter(b -> ServiceBindingUtils.matches(b, XSUAA));

		String xsuaaServiceBinding = runtime.getEnvironment().getCdsProperties().getSecurity().getXsuaa().getBinding();
		if (!StringUtils.isEmpty(xsuaaServiceBinding)) {
			xsuaaBindings = xsuaaBindings.filter(b -> xsuaaServiceBinding.equalsIgnoreCase(b.getName().get()));
		}

		return xsuaaBindings.collect(Collectors.toList());
	}

	/**
	 * @return	All found identity service (IAS) bindings. In case a specific service binding is chosen via cds.security.identity.binding,
	 * 	 * 		the list contains only this binding if available.
	 */
	public List<ServiceBinding> getIasServiceBindings() {
		// IAS-bindings unfortunately have no tags, but give UPSIs the chance to tag to "identity"
		Stream<ServiceBinding> iasBindings = runtime.getEnvironment().getServiceBindings().filter(b -> ServiceBindingUtils.matches(b, IDENTITY));

		String iasServiceBinding = runtime.getEnvironment().getCdsProperties().getSecurity().getIdentity().getBinding();
		if (!StringUtils.isEmpty(iasServiceBinding)) {
			iasBindings = iasBindings.filter(b -> iasServiceBinding.equalsIgnoreCase(b.getName().get()));
		}

		return iasBindings.collect(Collectors.toList());
	}

	/**
	 * There are three different kinds of CP Security authentication.
	 * Library spring-xsuaa supports XSUAA and IAS_XSUAA authentication (cds-feature-xsuaa required).
	 * Library spring-security supports IAS and XSUAA (cds-feature-identity required).
	 */
	public static enum AuthenticationType {
		XSUAA,		// XSUAA authentication + XSUAA users
		IAS_XSUAA,	// IAS authentication + XSUAA users
		IAS;		// IAS authentication + IAS users
	};

	/**
	 * @return	The authentication type configured by the application
	 */
	public AuthenticationType configuredAuthenticationType() {

		boolean xsuaaBindings = !getXsuaaServiceBindings().isEmpty();
		boolean iasBindings = !getIasServiceBindings().isEmpty();

		boolean isXsuaaEnabled = runtime.getEnvironment().getCdsProperties().getSecurity().getXsuaa().isEnabled();
		boolean isIasEnabled = runtime.getEnvironment().getCdsProperties().getSecurity().getIdentity().isEnabled();

		if (xsuaaBindings && isXsuaaEnabled) {
			return (iasBindings && isIasEnabled) ? AuthenticationType.IAS_XSUAA : AuthenticationType.XSUAA;

		} else if (iasBindings && isIasEnabled) {
			return AuthenticationType.IAS;
		} else {
			return null;
		}
	}
}
