/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAccessor;

public class TemporalRangeUtils {

	// smallest instant that worked in tests on a HANA DB
	public static final Instant MIN_VALID_FROM = Instant.parse("0001-01-01T00:00:00Z");
	public static final Instant MAX_VALID_TO = Instant.parse("9999-12-31T23:59:59.999999999Z");

	public static Instant[] getTemporalRanges(Instant validAt) {
		Instant[] range = new Instant[2];
		if (validAt == null) {
			// use now() if parameter valid-at isn't provided:
			// https://github.wdf.sap.corp/pages/cap/guides/temporal-data#reading-temporal-data
			range[0] = Instant.now();
		} else {
			range[0] = validAt;
		}
		range[1] = range[0].plusMillis(1);
		return truncatedRange(range);
	}

	public static Instant[] getTemporalRanges(String validFromQueryParam, String validToQueryParam, String validAtQueryParam) {

		Instant validFrom = parseDate(validFromQueryParam, QueryParameters.VALID_FROM);
		Instant validTo = parseDate(validToQueryParam, QueryParameters.VALID_TO);

		if (validFrom == null && validTo == null) {
			Instant validAt = parseDate(validAtQueryParam, QueryParameters.VALID_AT);
			return getTemporalRanges(validAt);
		}

		if (validFrom == null) {
			validFrom = MIN_VALID_FROM;
		}

		if (validTo == null) {
			validTo = MAX_VALID_TO;
		}

		return truncatedRange(new Instant[] {validFrom, validTo});
	}

	private static Instant parseDate(String date, String hint) {
		if (!StringUtils.isEmpty(date)) {
			try {
				// expects date-time in format: '2011-12-03T10:15:30' or
				// '2011-12-03T10:15:30+01:00'
				return Instant.from(DateTimeFormatter.ISO_OFFSET_DATE_TIME.parse(date));
			} catch (DateTimeParseException e1) {
				try {
					// expects just date in format: '2011-12-03' or '2011-12-03+01:00'
					TemporalAccessor accessor = DateTimeFormatter.ISO_DATE.parse(date);

					// extract zone offset from parsed date
					ZoneOffset zoneOffset = getZoneOffset(accessor);

					// since there is only a date, we need to guess the time according the hint
					LocalTime time = getLocalTime(hint);

					// add time to a have a complete date-time and get it converted to instant
					LocalDateTime dateTime = LocalDate.from(accessor).atTime(time);

					// use given time offset for creating instant
					return dateTime.toInstant(zoneOffset);
				} catch (DateTimeParseException e2) { // NOSONAR
					// try Timestamp format as fallback
					try {
						return Timestamp.valueOf(date).toLocalDateTime().toInstant(ZoneOffset.UTC);
					} catch (IllegalArgumentException e3) { // NOSONAR
						throw new ErrorStatusException(CdsErrorStatuses.INVALID_DATE_VALUE, hint, e1);
					}
				}
			}
		}
		return null;
	}

	private static ZoneOffset getZoneOffset(TemporalAccessor accessor) {
		if (accessor.isSupported(ChronoField.OFFSET_SECONDS)) {
			return ZoneOffset.ofTotalSeconds(accessor.get(ChronoField.OFFSET_SECONDS));
		}
		// return UTC if given temporal object doesn't provide an offset field
		return ZoneOffset.UTC;
	}

	private static LocalTime getLocalTime(String hint) {
		switch (hint) {
		case QueryParameters.VALID_TO:
			return LocalTime.MAX;
		case QueryParameters.VALID_FROM:
		case QueryParameters.VALID_AT:
		default:
			return LocalTime.MIN;
		}
	}

	private static Instant[] truncatedRange(Instant[] range) {
		Instant[] truncatedRange = new Instant[range.length];
		for(int i=0; i<range.length; ++i) {
			// truncate to micros, as timestamps in CAP have micro-seconds precision
			truncatedRange[i] = range[i].truncatedTo(ChronoUnit.MICROS);
		}
		return truncatedRange;
	}
}
