/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.datasource;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.sql.DataSource;

import com.sap.cds.services.datasource.DataSourceDescriptor;
import com.sap.cds.services.datasource.DataSourceDescriptorFactory;
import com.sap.cds.services.datasource.DataSourceFactory;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.ExtendedServiceLoader;

public class DataSourceUtils {

	public enum PoolType {
		HIKARI("hikari"),
		TOMCAT("tomcat"),
		DBCP2("dbcp2");

		private final String poolType;

		private PoolType(String poolType) {
			this.poolType = poolType;
		}

		@Override
		public String toString() {
			return poolType;
		}
	}

	/**
	 * @param runtime the {@link CdsRuntime}
	 *
	 * @return the list of available datasource descriptor objects
	 */
	public static List<DataSourceDescriptor> getDataSourceDescriptors(CdsRuntime runtime) {
		List<DataSourceDescriptor> descriptors = new ArrayList<>();
		ExtendedServiceLoader.loadAll(DataSourceDescriptorFactory.class, runtime)
				.forEachRemaining(p -> descriptors.addAll(p.create()));
		return descriptors;
	}

	/**
	 * Returns the available datasource objects as a map.
	 * The key in the map is the name of the datasource.
	 *
	 * @param runtime the {@link CdsRuntime}
	 *
	 * @return the map of datasources
	 */
	public static Map<String, DataSource> getDataSources(CdsRuntime runtime) {
		Map<String, DataSource> dataSources = new HashMap<>();
		ExtendedServiceLoader.loadAll(DataSourceFactory.class, runtime)
				.forEachRemaining(p -> dataSources.putAll(p.create()));
		return dataSources;
	}

	public static String getDataSourceSection(String name, PoolType poolType) {
		return "cds.data-source.%s.%s".formatted(toKebabCase(name), poolType);
	}

	private static String toKebabCase(String str) {
		return str.toLowerCase(Locale.US).replace("_", "-");
	}

}
