/*******************************************************************************
 *   © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/

package com.sap.cds.services.utils.lib.tools.impl;


import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

public class Retry {
	public static final WaitTimeFunction EXP_WAIT_TIME_FUNCTION = (time, numberOfRetries) -> Duration.ofMillis(((Double) (time.toMillis() * Math.pow(2, numberOfRetries))).longValue());
	public static final WaitTimeFunction CONSTANT_WAIT_TIME_FUNCTION = (time, numberOfRetries) -> time;
	public static final WaitTimeFunction LINEAR_WAIT_TIME_FUNCTION = (time, numberOfRetries) -> Duration.ofMillis(time.toMillis() * (numberOfRetries + 1));
	private final int numOfRetries;
	private final List<Class> retryExceptions = new ArrayList<>();//NOSONAR
	private final Duration baseWaitTime;
	private final WaitTimeFunction waitTimeFunction;

	private Retry(int numOfRetries, List<Class<? extends Throwable>> retryExceptions, Duration baseWaitTime,
				  WaitTimeFunction waitTimeFunction) {
		if (waitTimeFunction == null) {
			waitTimeFunction = CONSTANT_WAIT_TIME_FUNCTION;
		}
		this.numOfRetries = numOfRetries;
		this.retryExceptions.addAll(retryExceptions);
		if (baseWaitTime != null) {
			this.baseWaitTime = baseWaitTime;
		} else {
			this.baseWaitTime = Duration.ofSeconds(1);
		}
		this.waitTimeFunction = waitTimeFunction;
	}

	public <T> T execute(SupplierWithException<T> supplier) throws Exception {
		int retryRun = -1;
		do {
			retryRun++;
			try {
				return supplier.get();
			} catch (Throwable e) {//NOSONAR
				if (retryRun >= numOfRetries || !isARetryException(e)) {
					throw e;
				}
				if (e instanceof WaitTimeSupplier waitTimeSupplier && waitTimeSupplier.getWaitTime() != null) {
					var requestedWaitTime = waitTimeSupplier.getWaitTime().toMillis();
					if (requestedWaitTime > getMaximumWaitTime()) {
						throw e;
					}
					sleep(retryRun, waitTimeSupplier.getWaitTime());
				} else {
					sleep(retryRun, null);
				}
			}
		} while (true);
	}

	public void execute(VoidConsumerWithException consumer) throws Exception {
		execute(() -> {
			consumer.accept();
			return null;
		});
	}

	private void sleep(int retryRun, Duration requestedWaitTime) {
		try {
			if (requestedWaitTime != null) {
				Thread.sleep(requestedWaitTime.toMillis());
			} else {
				Thread.sleep(waitTimeFunction.getWaitTime(baseWaitTime, retryRun).toMillis());
			}
		} catch (InterruptedException interruptedException) {
			Thread.currentThread().interrupt();
		}
	}

	private boolean isARetryException(Throwable e) {
		return retryExceptions.stream().anyMatch(clazz -> clazz.isInstance(e));
	}

	private long getMaximumWaitTime() {
		long maxWaitTime = 0;
		for (int i = 0; i < numOfRetries; i++) {
			maxWaitTime += waitTimeFunction.getWaitTime(baseWaitTime, i).toMillis();
		}
		return maxWaitTime;
	}

	public static final class RetryBuilder {
		private int numOfRetries;
		private List<Class<? extends Throwable>> retryExceptions = new ArrayList<>();//NOSONAR
		private Duration baseWaitTime;
		private WaitTimeFunction waitTimeFunction;

		private RetryBuilder() {
		}

		public static RetryBuilder create() { //NOSONAR
			return new RetryBuilder();
		}

		public RetryBuilder numOfRetries(int numOfRetries) {
			this.numOfRetries = numOfRetries;
			return this;
		}

		public RetryBuilder retryExceptions(Class<? extends Throwable>... retryExceptions) {
			this.retryExceptions = Arrays.asList(retryExceptions);
			return this;
		}

		public RetryBuilder retryExceptions(Set<Class<? extends Throwable>> retryExceptions) {
			if (retryExceptions != null) {
				this.retryExceptions.addAll(retryExceptions);
			}
			return this;
		}

		public RetryBuilder baseWaitTime(Duration baseWaitTime) {
			this.baseWaitTime = baseWaitTime;
			return this;
		}

		public RetryBuilder waitTimeFunction(WaitTimeFunction waitTimeFunction) {
			this.waitTimeFunction = waitTimeFunction;
			return this;
		}

		public Retry build() {
			return new Retry(numOfRetries, retryExceptions, baseWaitTime, waitTimeFunction);
		}
	}
}


