/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cds.services.utils.cert;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertValidator {
    private final Logger logger = LoggerFactory.getLogger(CertValidator.class);
    private static final String CERTIFICATE_HEADER = "-----BEGIN CERTIFICATE-----";
    private static final String CERTIFICATE_HEADER_ENCODED = "-----BEGIN%20CERTIFICATE-----";
    private static final String CALLBACK_CERTIFICATE_ISSUER = "callback_certificate_issuer";
    private static final String CALLBACK_CERTIFICATE_SUBJECT = "callback_certificate_subject";
    private static final String X_509 = "X.509";
    private static final ObjectMapper mapper = new ObjectMapper();
    private static final String XFCC_HEADER_CERT_KEY = "cert=";
    private final CertificateFactory certFactory = CertValidator.getX509CertFactory();
    private final Map<String, Object> expectedIssuer;
    private final Map<String, Object> expectedSubject;
    private final String clientCertificateHeaderName;

    CertValidator(String clientCertificateHeaderName, Map<String, Object> expectedIssuer, Map<String, Object> expectedSubject) {
        this.clientCertificateHeaderName = clientCertificateHeaderName;
        this.expectedIssuer = expectedIssuer;
        this.expectedSubject = expectedSubject;
    }

    public static CertValidator create(String clientCertificateHeaderName, ServiceBinding binding) {
        try {
            Map<String, Object> expectedIssuer = CertValidator.jsonStrToFlatMap((String)binding.getCredentials().get(CALLBACK_CERTIFICATE_ISSUER));
            Map<String, Object> expectedSubject = CertValidator.jsonStrToFlatMap((String)binding.getCredentials().get(CALLBACK_CERTIFICATE_SUBJECT));
            return new CertValidator(clientCertificateHeaderName, expectedIssuer, expectedSubject);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static CertValidator create(String clientCertificateHeaderName, String expectedIssuer, String expectedSubject) {
        return new CertValidator(clientCertificateHeaderName, CertValidator.strToFlatMap(expectedIssuer), CertValidator.strToFlatMap(expectedSubject));
    }

    private static CertificateFactory getX509CertFactory() {
        try {
            return CertificateFactory.getInstance(X_509);
        }
        catch (CertificateException e) {
            throw new ServiceException("No X.509 certificate provider found", new Object[]{e});
        }
    }

    @VisibleForTesting
    X509Certificate getX509Certificate(String cert) throws Exception {
        ByteArrayInputStream is;
        if (cert.startsWith(CERTIFICATE_HEADER)) {
            this.logger.debug("Certificate starts with {}", (Object)CERTIFICATE_HEADER);
            is = new ByteArrayInputStream(cert.getBytes(StandardCharsets.UTF_8));
        } else if (cert.startsWith(CERTIFICATE_HEADER_ENCODED)) {
            this.logger.debug("Certificate starts with {}", (Object)CERTIFICATE_HEADER_ENCODED);
            is = new ByteArrayInputStream(URLDecoder.decode(cert, StandardCharsets.UTF_8).getBytes(StandardCharsets.UTF_8));
        } else {
            this.logger.debug("Certificate is base64 encoded");
            is = new ByteArrayInputStream(Base64.getDecoder().decode(cert));
        }
        return (X509Certificate)this.certFactory.generateCertificate(is);
    }

    public void validateCertFromRequestContext(RequestContext requestContext) {
        String cert = requestContext.getParameterInfo().getHeader(this.clientCertificateHeaderName);
        if (cert != null) {
            X509Certificate certificate;
            String[] parts = cert.split(";");
            if (parts.length > 1) {
                cert = Arrays.stream(parts).filter(p -> p.toLowerCase().startsWith(XFCC_HEADER_CERT_KEY)).map(p -> p.substring(XFCC_HEADER_CERT_KEY.length()).replace("\"", "")).findFirst().orElseThrow(() -> {
                    this.logger.debug("No key 'cert' found in header '{}' that includes a certificate.", (Object)this.clientCertificateHeaderName);
                    return new ErrorStatusException((ErrorStatus)ErrorStatuses.UNAUTHORIZED, new Object[0]);
                });
            }
            try {
                certificate = this.getX509Certificate(cert);
            }
            catch (Exception e) {
                this.logger.debug("Header '{}' did not include a valid certificate.", (Object)this.clientCertificateHeaderName);
                throw new ErrorStatusException((ErrorStatus)ErrorStatuses.UNAUTHORIZED, new Object[0]);
            }
            if (!this.isValidCertIssuer(certificate.getIssuerX500Principal().toString()) || !this.isValidCertSubject(certificate.getSubjectX500Principal().toString())) {
                this.logger.debug("Subject or issuer of certificate from header '{}' not as expected.", (Object)this.clientCertificateHeaderName);
                throw new ErrorStatusException((ErrorStatus)ErrorStatuses.FORBIDDEN, new Object[0]);
            }
        } else {
            this.logger.debug("Request did not provide header '{}'.", (Object)this.clientCertificateHeaderName);
            throw new ErrorStatusException((ErrorStatus)ErrorStatuses.UNAUTHORIZED, new Object[0]);
        }
    }

    @VisibleForTesting
    boolean isValidCertSubject(String subjectFromCert) {
        boolean valid;
        boolean bl = valid = subjectFromCert.contains("C=" + this.expectedSubject.get("C")) && subjectFromCert.contains("CN=" + this.expectedSubject.get("CN")) && ("*".equals(this.expectedSubject.get("L")) || subjectFromCert.contains("L=" + this.expectedSubject.get("L"))) && subjectFromCert.contains("O=" + this.expectedSubject.get("O"));
        if (this.expectedSubject.get("OU").toString().startsWith("[") && this.expectedSubject.get("OU").toString().endsWith("]")) {
            ArrayList items = (ArrayList)this.expectedSubject.get("OU");
            for (String item : items) {
                valid = valid && subjectFromCert.contains("OU=" + item);
            }
        } else {
            valid = valid && subjectFromCert.contains("OU=" + this.expectedSubject.get("OU"));
        }
        return valid;
    }

    @VisibleForTesting
    boolean isValidCertIssuer(String issuerFromCert) {
        return issuerFromCert.contains("C=" + this.expectedIssuer.get("C")) && issuerFromCert.contains("OU=" + this.expectedIssuer.get("OU")) && issuerFromCert.contains("CN=" + this.expectedIssuer.get("CN")) && ("*".equals(this.expectedIssuer.get("L")) || issuerFromCert.contains("L=" + this.expectedIssuer.get("L"))) && issuerFromCert.contains("O=" + this.expectedIssuer.get("O"));
    }

    @VisibleForTesting
    static Map<String, Object> jsonStrToFlatMap(String json) throws IOException {
        return (Map)mapper.readValue(json, (TypeReference)new TypeReference<HashMap<String, Object>>(){});
    }

    @VisibleForTesting
    static Map<String, Object> strToFlatMap(String input) {
        return Arrays.stream(input.split(",")).map(pair -> pair.split("=")).filter(keyValue -> ((String[])keyValue).length == 2).collect(Collectors.toMap(keyValue -> keyValue[0].trim(), keyValue -> keyValue[1].trim(), (existing, replacement) -> {
            if (existing instanceof List) {
                ((List)existing).add((String)replacement);
                return existing;
            }
            return new ArrayList<String>(Arrays.asList((String)existing, (String)replacement));
        }, HashMap::new));
    }
}

