/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ETagUtils {
  private static Logger log = LoggerFactory.getLogger(ETagUtils.class);
  private static final String NULL_ETAG = "W/\"\"";

  private static final String WEAK_PATTERN = "W/\"(.*)\"";
  private static final String STRONG_PATTERN = "\"(.*)\"";
  private static final Pattern strongPattern = Pattern.compile(STRONG_PATTERN);
  private static final Pattern weakPattern = Pattern.compile(WEAK_PATTERN);
  private static final String DELIMITER = ",";
  private static final String SPACE = "\\s*";

  /**
   * @param headerValue headerValue
   * @return extract plain string value from ETag value
   */
  public static String extractETagValue(String headerValue) {
    log.debug("Extracting eTag header value {}.", headerValue);

    Matcher weakMatcher = weakPattern.matcher(headerValue);
    Matcher strongMatcher = strongPattern.matcher(headerValue);
    if (weakMatcher.matches()) {
      return weakMatcher.group(1);
    } else if (strongMatcher.matches()) {
      return strongMatcher.group(1);
    } else {
      return headerValue;
    }
  }

  /**
   * @param headerValues headerValues
   * @return extract plain string values from ETag values and returns as a List
   */
  public static List<String> parseHeaderValues(List<String> headerValues) {
    List<String> headerValueList = new ArrayList<>();
    if (headerValues != null) {
      headerValueList =
          headerValues.stream().map(ETagUtils::extractETagValue).collect(Collectors.toList());
    }
    return ((headerValueList.isEmpty()) ? null : headerValueList);
  }

  /**
   * @param propValue propValue
   * @return returns etag value for response
   */
  public static String createETagHeaderValue(String propValue) {
    // Only weak ETag value will be returned in response
    String eTag = NULL_ETAG;
    if (propValue != null) {
      eTag = "W/\"" + propValue + "\"";
    }
    return eTag;
  }

  /**
   * @param header header
   * @return parse and return the list of header
   */
  public static List<String> parseHeader(String header) {
    List<String> headerValueList = new ArrayList<>();
    if (header != null && !header.isEmpty()) {
      String[] arr = header.trim().split(SPACE + DELIMITER + SPACE);
      headerValueList.addAll(parseHeaderValues(Arrays.asList(arr)));
    }
    return ((headerValueList.isEmpty()) ? null : headerValueList);
  }
}
