/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils;

import com.sap.cds.ql.cqn.CqnComparisonPredicate;
import com.sap.cds.ql.cqn.CqnComparisonPredicate.Operator;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnLiteral;
import com.sap.cds.ql.cqn.CqnNullValue;
import com.sap.cds.ql.cqn.CqnPlain;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.services.messages.MessageTarget;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

/** A utility class providing OData related helper methods. */
public class ODataUtils {

  /**
   * Converts a given CDS entity name into an OData entity name.
   *
   * @param cdsEntityName the CDS entity name
   * @return the OData entity name
   */
  public static String toODataName(String cdsEntityName) {
    return cdsEntityName.replace(".", "_");
  }

  /**
   * Converts the given parameters and {@link CqnReference} to an OData V2 or V4 target string
   *
   * @param bindingParameter the binding parameter of the action or function, or {@code null}
   * @param parameter the parameter of the {@link MessageTarget}, or {@code null}
   * @param ref the {@link CqnReference} of the {@link MessageTarget}, or {@code null}
   * @param v2 {@code false}, if the target string is for OData V4, {@code true} if it is for OData
   *     V2
   * @return the OData target string
   */
  public static String toODataTarget(
      String bindingParameter, String parameter, CqnReference ref, boolean v2) {
    List<? extends Segment> segments = ref == null ? Collections.emptyList() : ref.segments();

    if (segments.isEmpty()) {
      return parameter;
    }

    String prefix;
    if (MessageTarget.PARAMETER_CQN.equals(parameter)) {
      prefix = bindingParameter;
    } else {
      prefix = parameter;
    }

    StringBuilder target = new StringBuilder(StringUtils.isEmpty(prefix) ? "" : prefix);

    if (!segments.isEmpty() && target.length() > 0 && target.charAt(target.length() - 1) != '/') {
      target.append("/");
    }

    for (Segment segment : segments) {
      target.append(segment.id());
      if (segment.filter().isPresent()) {
        target.append("(");
        AtomicInteger beforeIsActiveEntityPosition = new AtomicInteger(-1);
        segment
            .filter()
            .get()
            .accept(
                new CqnVisitor() {

                  @Override
                  public void visit(CqnComparisonPredicate predicate) {
                    if (predicate.operator() == Operator.EQ
                        || predicate.operator() == Operator.IS) {
                      CqnValue left = predicate.left();
                      CqnValue right = predicate.right();
                      CqnValue value = null;
                      String key = null;
                      if (left instanceof CqnElementRef ref) {
                        key = ref.firstSegment();
                        value = right;
                      } else if (right instanceof CqnElementRef ref) {
                        key = ref.firstSegment();
                        value = left;
                      }
                      if (key != null) {
                        Object theValue = null;
                        if (value instanceof CqnLiteral<?> literal) {
                          theValue = literal.value();
                          if (theValue instanceof String string) {
                            if (!isUUID(string)) {
                              theValue = "'" + theValue + "'";
                            } else if (v2) {
                              theValue = "guid" + "'" + theValue + "'";
                            }
                          }
                        } else if (value instanceof CqnNullValue) {
                          theValue = "null";
                        } else if (value instanceof CqnPlain plain) {
                          theValue = plain.plain();
                        }
                        if (theValue != null) {
                          String toAppend = key + "=" + theValue + ",";
                          // TODO dirty workaround for Fiori requiring certain key order
                          // will only work for single key or single key draft enabled entities
                          int insertAt = beforeIsActiveEntityPosition.get();
                          if (insertAt > -1) {
                            target.insert(insertAt, toAppend);
                            beforeIsActiveEntityPosition.set(insertAt + toAppend.length());
                          } else {
                            if (key.equals("IsActiveEntity")) {
                              beforeIsActiveEntityPosition.set(target.length());
                            }
                            target.append(toAppend);
                          }
                        }
                      }
                    }
                  }
                });
        target.deleteCharAt(target.length() - 1);
        target.append(")");
      }
      target.append("/");
    }

    if (!segments.isEmpty()) {
      target.deleteCharAt(target.length() - 1);
    }

    return target.toString();
  }

  // TODO this check should actually be done against the type in the model
  private static boolean isUUID(String s) {
    try {
      UUID.fromString(s);
      return true;
    } catch (IllegalArgumentException e) { // NOSONAR
      return false;
    }
  }
}
