/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.stream.Collectors;

/** Helper functions providing simple {@link String} operations */
public class StringUtils {

  private StringUtils() {
    // empty
  }

  /**
   * Tests if a {@link String} has content (e.g. none space characters) or not.
   *
   * @param value The string to be tested
   * @return {@code true} in case {@code value} is {@code null} or has no content.
   */
  public static boolean isEmpty(String value) {
    return value == null || value.trim().length() == 0;
  }

  /**
   * Returns the first not empty {@code String} from the parameters or {@code null} if not existing.
   *
   * @param values The {@code String} values in sorted order
   * @return The first not empty {@code String} or {@code null}
   */
  public static String notEmpty(String... values) {
    if (values != null) {
      for (String value : values) {
        if (!isEmpty(value)) {
          return value;
        }
      }
    }

    return null;
  }

  /**
   * Trims a string from a certain character
   *
   * @param value the string
   * @param c the character
   * @return the trimmed string
   */
  public static String trim(String value, char c) {
    int len = value.length();
    int st = 0;
    char[] val = value.toCharArray();

    while ((st < len) && (val[st] == c)) {
      st++;
    }
    while ((st < len) && (val[len - 1] == c)) {
      len--;
    }

    return ((st > 0) || (len < value.length())) ? value.substring(st, len) : value;
  }

  /**
   * Compares two string property maps. The keys are compared case-insensitive. Values are compared
   * case-sensitive.
   *
   * @param props map to be compared
   * @param otherProps map to be compared
   * @return <code>true</code> id equals and <code>false</code> otherwise
   */
  public static boolean equalsMapIgnoreCase(
      Map<String, String> props, Map<String, String> otherProps) {

    if (props == null && otherProps == null) {
      return true;
    }

    if (props == null || otherProps == null || props.size() != otherProps.size()) {
      return false;
    }

    for (Entry<String, String> prop : props.entrySet()) {
      if (!findEqualsKeyValue(prop.getKey(), prop.getValue(), otherProps)) {
        return false;
      }
    }

    return true;
  }

  private static boolean findEqualsKeyValue(String key, String value, Map<String, String> props) {

    for (Entry<String, String> prop : props.entrySet()) {
      // find equals key
      if (key == prop.getKey() || (key != null && key.equalsIgnoreCase(prop.getKey()))) {
        // check the value
        if (Objects.equals(value, prop.getValue())) {
          return true;
        }
      }
    }

    return false;
  }

  /**
   * Converts a map of key value pairs to a string 'key1=value1, key2=value2, ...'
   *
   * @param keyValueMap map containing key value pairs
   * @return string representation
   */
  public static String stringifyMap(Map<String, Object> keyValueMap) {
    return keyValueMap.entrySet().stream()
        .map(k -> k.getKey() + "=" + k.getValue())
        .collect(Collectors.joining(", "));
  }

  /**
   * Converts a list of values to a string 'value1, value2, ...'
   *
   * @param values list of values to stringify
   * @return string representation
   */
  public static String stringifyList(List<String> values) {
    return values.stream().collect(Collectors.joining(", "));
  }

  /**
   * Converts an array of objects into a string, while catching and handling all exception types.
   *
   * @param args the array of objects
   * @return the stringified objects
   */
  public static String toStringSafely(Object[] args) {
    if (args != null) {
      StringBuffer sb = new StringBuffer();
      for (Object arg : args) {
        sb.append('<');
        try {
          if (arg != null) {
            sb.append(arg.toString());
          } else {
            sb.append("null");
          }
        } catch (Exception e) { // NOSONAR
          sb.append(e.getMessage());
        }
        sb.append('>');
      }
      return sb.toString();
    }
    return "<null>";
  }
}
