/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.environment;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cloud.environment.servicebinding.api.DefaultServiceBinding;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Parser for Open Service Broker API service bindings */
@JsonIgnoreProperties(ignoreUnknown = true)
public class VcapServicesParser {

  public static final String VCAP_SERVICES = "VCAP_SERVICES";

  private static final Logger logger = LoggerFactory.getLogger(VcapServicesParser.class);
  private static final ObjectMapper mapper = new ObjectMapper();

  public static List<ServiceBinding> getServiceBindings(String vcapServices) {
    List<ServiceBinding> bindings = new ArrayList<>();

    if (vcapServices != null && vcapServices.trim().length() > 0) {
      try {
        JsonNode vcapNodes = mapper.readTree(vcapServices);

        // if the VCAP_SERVICES key is part of the JSON, use its value
        if (vcapNodes.has(VCAP_SERVICES)) {
          vcapNodes = vcapNodes.get(VCAP_SERVICES);
        }

        // for each service a field exists
        vcapNodes
            .properties()
            .forEach(
                serviceField -> {

                  // for each binding an entry in the array exists
                  Iterator<JsonNode> bindingNodes = serviceField.getValue().elements();
                  int i = 0;
                  while (bindingNodes.hasNext()) {
                    try {
                      VcapServicesParser parseBinding =
                          mapper.treeToValue(bindingNodes.next(), VcapServicesParser.class);
                      ServiceBinding binding =
                          DefaultServiceBinding.builder()
                              .copy(Collections.emptyMap())
                              .withName(parseBinding.getName())
                              .withServiceName(serviceField.getKey())
                              .withServicePlan(parseBinding.servicePlan)
                              .withTags(
                                  parseBinding.tags != null
                                      ? parseBinding.tags
                                      : Collections.emptyList())
                              .withCredentials(
                                  parseBinding.credentials != null
                                      ? parseBinding.credentials
                                      : Collections.emptyMap())
                              .build();
                      bindings.add(binding);
                    } catch (IOException e) {
                      logger.warn(
                          "Could not parse service binding of service '{}' at index '{}'.",
                          serviceField.getKey(),
                          i,
                          e);
                    }
                    ++i;
                  }
                });
      } catch (IOException e) {
        logger.warn("Could not parse " + VCAP_SERVICES, e);
      }
    }

    return bindings;
  }

  @JsonProperty("name")
  private String name;

  @JsonProperty("binding_name")
  private String bindingName;

  @JsonProperty("instance_name")
  private String instanceName;

  @JsonProperty("plan")
  private String servicePlan;

  @JsonProperty("tags")
  private List<String> tags;

  @JsonProperty("credentials")
  private Map<String, Object> credentials;

  private String getName() {
    if (name != null && !name.isEmpty()) {
      return name;
    } else if (bindingName != null && !bindingName.isEmpty()) {
      return bindingName;
    } else if (instanceName != null && !instanceName.isEmpty()) {
      return instanceName;
    }
    return null;
  }
}
