/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.hcql;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.sap.cds.impl.parser.StructDataParser;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.util.CdsTypeUtils;
import java.util.Optional;

public class HcqlUtils {

  private HcqlUtils() {
    // avoid instances
  }

  public static Object convertToType(CdsType type, ObjectNode node, String name) {
    if (type.isSimple()) {
      CdsBaseType baseType = type.as(CdsSimpleType.class).getType();
      Optional<String> fieldValue = getValueAsString(node, name);
      if (fieldValue.isPresent()) {
        return CdsTypeUtils.parse(baseType, fieldValue.get());
      } else {
        return null;
      }
    } else if (type.isStructured()) {
      return StructDataParser.parseObjectOf(
          type.as(CdsStructuredType.class), (ObjectNode) node.get(name));
    } else if (type.isArrayed()) {
      CdsType itemsType = type.as(CdsArrayedType.class).getItemsType();
      if (itemsType.isSimple()) {
        return StructDataParser.parseArrayOf(
            itemsType.as(CdsSimpleType.class), (ArrayNode) node.get(name));
      } else if (itemsType.isStructured()) {
        return StructDataParser.parseArrayOf(
            itemsType.as(CdsStructuredType.class), (ArrayNode) node.get(name));
      }
    }
    return null;
  }

  public static String getValueAsString(ObjectNode node, String name, String defaultValue) {
    return getValueAsString(node, name).orElse(defaultValue);
  }

  public static Optional<String> getValueAsString(ObjectNode node, String name) {
    JsonNode child = node.get(name);
    // child could be not null, but a NullNode
    if (child != null && !child.isNull()) {
      return Optional.of(child.isValueNode() ? child.asText() : child.toString());
    }
    return Optional.empty();
  }
}
