/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.lib.mt;

import com.google.common.base.Ticker;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.sap.cds.CdsDataStoreConnector;
import com.sap.cds.CdsException;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.utils.lib.mtx.CdsDataStoreConnectorCreator;
import com.sap.cds.services.utils.lib.mtx.MetaDataAccessor;
import com.sap.cds.services.utils.lib.mtx.impl.CacheParams;
import java.util.function.BiPredicate;
import java.util.function.Function;

/**
 * Responsible for retrieval of {@code CdsDataStoreConnectors}. For performance reasons {@code
 * CdsDataStore} connector instances are cached per tenant. The cache is synchronized with the
 * {@code MetaDataAccessor}'s cache: cache {@code CdsDataStore} entries are automatically refreshed
 * when a cached model changes in the {@code MetaDataAccessor}'s cache. To manually refresh a {@code
 * CdsDataStore} for a particular tenant the tenant's model must be refreshed or evicted at the
 * {@code MetaDataAccessor}.
 *
 * @see MetaDataAccessor#refresh(String)
 * @see MetaDataAccessor#evict(String)
 */
public class CdsDataStoreLookup {
  private final BiPredicate<String, CdsModel> isModelOutDated;
  private final LoadingCache<String, CdsDataStoreConnector> tenantToConnector;

  /**
   * @param cdsDataStoreConnectorCreator factory that creates a {@link CdsDataStoreConnector}
   * @param isModelOutDated predicate that takes the tenant id and the current model as input and
   *     decides if the model is out dated
   * @param cacheParams Parameters that control cache lifecycle
   * @param cacheTicker Optional ticker used by guava cache for testing purposes, use null for
   *     productive use
   */
  public CdsDataStoreLookup(
      CdsDataStoreConnectorCreator cdsDataStoreConnectorCreator,
      BiPredicate<String, CdsModel> isModelOutDated,
      CacheParams cacheParams,
      Ticker cacheTicker) {
    this.isModelOutDated = isModelOutDated;
    this.tenantToConnector =
        buildCache(cacheParams, cacheTicker, cdsDataStoreConnectorCreator::create);
  }

  /**
   * Determine a data store connector for a tenant
   *
   * @param tenantId tenant identifier
   * @return a CDS data store connector
   */
  public CdsDataStoreConnector getCdsDataStoreConnector(String tenantId) throws CdsException {
    try {
      return tenantToConnector.getUnchecked(tenantId);
    } catch (UncheckedExecutionException e) {
      throw new CdsException(e);
    }
  }

  public void evictIfOutDated(String tenantId) {
    CdsDataStoreConnector connector = getCdsDataStoreConnector(tenantId);
    if (connector != null) {
      CdsModel cdsModel = connector.reflect();
      if (isModelOutDated.test(tenantId, cdsModel)) {
        tenantToConnector.invalidate(tenantId);
      }
    }
  }

  private static LoadingCache<String, CdsDataStoreConnector> buildCache(
      CacheParams params, Ticker cacheTicker, Function<String, CdsDataStoreConnector> loader) {
    CacheBuilder<Object, Object> builder =
        CacheBuilder.newBuilder()
            .maximumSize(params.getMaximumSize())
            .expireAfterAccess(params.getExpirationDuration(), params.getExpirationDurationUnit());
    if (cacheTicker != null) {
      builder.ticker(cacheTicker);
    }
    return builder.build(
        new CacheLoader<String, CdsDataStoreConnector>() {
          @Override
          public CdsDataStoreConnector load(String key) {
            return loader.apply(key);
          }
        });
  }
}
