/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.lib.mtx.impl;

import com.sap.cds.CdsException;
import java.time.Duration;
import java.util.concurrent.TimeUnit;

/** Value class that stores cache parameters for {@link MetaDataAccessorImpl} */
public class CacheParams {
  private final long maximumSize;
  private final long expirationDuration;
  private final TimeUnit expirationDurationUnit;
  private final long refreshDuration;
  private final TimeUnit refreshDurationUnit;
  private final boolean synchronousRefresh;
  private final boolean withBaseModelETag;

  /**
   * @param maximumSize maximum size of cache for csn and edmx model
   * @param expirationDuration expiration duration unit after which model is deleted from cache
   *     after creation or last refresh
   * @param refreshDuration refresh duration unit after which model is reread from sidecar
   * @param synchronousRefresh cache is refreshed synchronously
   */
  public CacheParams(
      long maximumSize,
      Duration expirationDuration,
      Duration refreshDuration,
      boolean synchronousRefresh) {
    this(maximumSize, expirationDuration, refreshDuration, synchronousRefresh, false);
  }

  /**
   * @param maximumSize maximum size of cache for csn and edmx model
   * @param expirationDuration expiration duration unit after which model is deleted from cache
   *     after creation or last refresh
   * @param refreshDuration refresh duration unit after which model is reread from sidecar
   * @param synchronousRefresh cache is refreshed synchronously
   * @param withBaseModelETag determines whether the initial requests are tagged with base model
   *     etag
   */
  public CacheParams(
      long maximumSize,
      Duration expirationDuration,
      Duration refreshDuration,
      boolean synchronousRefresh,
      boolean withBaseModelETag) {
    this.maximumSize = maximumSize;
    this.expirationDuration = expirationDuration.getSeconds();
    this.expirationDurationUnit = TimeUnit.SECONDS;
    this.refreshDuration = refreshDuration.getSeconds();
    this.refreshDurationUnit = TimeUnit.SECONDS;
    this.synchronousRefresh = synchronousRefresh;
    this.withBaseModelETag = withBaseModelETag;
    validate();
  }

  private void validate() {
    if (maximumSize < 0)
      throw new CdsException("Maximum cache size must be greater than or equal to 0");
    if (expirationDuration < 0) throw new CdsException("Expiration duration must be positive");
    if (expirationDurationUnit == null)
      throw new CdsException("Expiration duration unit must not be null");
    if (refreshDuration <= 0) throw new CdsException("Refresh duration must be greater than 0");
    if (refreshDurationUnit == null)
      throw new CdsException("Refresh duration unit must not be null");
  }

  /**
   * @return maximum cache size
   */
  public long getMaximumSize() {
    return maximumSize;
  }

  /**
   * @return time after which cache entry expires
   */
  public long getExpirationDuration() {
    return expirationDuration;
  }

  /**
   * @return time unit for time after which cache entry expires
   */
  public TimeUnit getExpirationDurationUnit() {
    return expirationDurationUnit;
  }

  /**
   * @return time after which model is reread
   */
  public long getRefreshDuration() {
    return refreshDuration;
  }

  /**
   * @return time unit for time after which model is reread
   */
  public TimeUnit getRefreshDurationUnit() {
    return refreshDurationUnit;
  }

  public boolean isSynchronousRefresh() {
    return synchronousRefresh;
  }

  public boolean isWithBaseModelETag() {
    return withBaseModelETag;
  }
}
