/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.lib.tools.api;

import java.util.Optional;

/** Defines how a service call reacts to a response code */
@FunctionalInterface
public interface ResponseChecker {
  public static final ResponseChecker NONE = (int code) -> CheckResult.NO_PROBLEM;

  /**
   * This function controls how the service class reacts to response codes. If the response check
   * function identifies a code that indicates a situation not covered by the service contract, it
   * puts an exception object into the result object. In addition, flag isTemporaryProblem can be
   * set to indicate a problem that can be solved with a retry.
   *
   * @param code returned from http request
   * @return check result
   */
  public CheckResult checkCode(int code);

  public static class CheckResult {
    private final boolean isTemporaryProblem;
    private final Optional<Exception> exception;
    public static final CheckResult NO_PROBLEM = new CheckResult(null, false);

    /**
     * @param isTemporaryProblem is set to true to indicate a problem that can be solved with a
     *     retry
     * @param exception application chosen exception that will be passed to service caller
     */
    public CheckResult(Exception exception, boolean isTemporaryProblem) {
      this.isTemporaryProblem = isTemporaryProblem;
      this.exception = Optional.ofNullable(exception);
    }

    public Optional<Exception> getException() {
      return exception;
    }

    public boolean isTemporaryProblem() {
      return isTemporaryProblem;
    }
  }
}
