/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.lib.tools.impl;

import com.sap.cds.services.utils.lib.tools.api.HttpMethod;
import com.sap.cds.services.utils.lib.tools.api.QueryParameter;
import com.sap.cds.services.utils.lib.tools.api.RequestEnhancer;
import com.sap.cds.services.utils.lib.tools.api.ServiceCall;
import com.sap.cds.services.utils.lib.tools.api.ServiceCallMediators.End;
import com.sap.cds.services.utils.lib.tools.api.ServiceCallMediators.FromCreate;
import com.sap.cds.services.utils.lib.tools.api.ServiceCallMediators.FromHttp;
import com.sap.cds.services.utils.lib.tools.api.ServiceCallMediators.FromHttpMethod;
import com.sap.cds.services.utils.lib.tools.api.ServiceCallMediators.FromPathParameter;
import com.sap.cds.services.utils.lib.tools.api.ServiceCallMediators.FromPayload;
import com.sap.cds.services.utils.lib.tools.api.ServiceCallMediators.FromQuery;
import com.sap.cds.services.utils.lib.tools.api.ServiceCallMediators.FromRequestEnhancer;
import com.sap.cds.services.utils.lib.tools.api.ServiceEndpoint;
import com.sap.cds.services.utils.lib.tools.exception.InternalException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang3.StringUtils;

/** Use this builder to create an instance of the service abstraction. */
public final class ServiceCallBuilder
    implements FromCreate,
        FromHttp,
        FromHttpMethod,
        FromPayload,
        FromPathParameter,
        FromQuery,
        FromRequestEnhancer {
  private ServiceEndpoint serviceEndpoint = null;
  private final Map<String, String> headerFields = new HashMap<>();
  private final List<QueryParameter> queryParameters = new ArrayList<>();
  private Object payload = null;
  private List<String> pathParameters = new ArrayList<>();
  private HttpMethod httpMethod;
  private RequestEnhancer requestEnhancer = null;

  private ServiceCallBuilder(ServiceEndpoint serviceEndpoint) {
    this.serviceEndpoint = serviceEndpoint;
  }

  @Override
  public FromHttp http() {
    return this;
  }

  @Override
  public FromHttpMethod get() {
    this.httpMethod = HttpMethod.GET;
    return this;
  }

  @Override
  public FromHttpMethod put() {
    this.httpMethod = HttpMethod.PUT;
    return this;
  }

  @Override
  public FromHttpMethod post() {
    this.httpMethod = HttpMethod.POST;
    return this;
  }

  @Override
  public FromHttpMethod delete() {
    this.httpMethod = HttpMethod.DELETE;
    return this;
  }

  @Override
  public FromHttpMethod patch() {
    this.httpMethod = HttpMethod.PATCH;
    return this;
  }

  @Override
  public <P> FromPayload payload(P payload) {
    this.payload = payload;
    return this;
  }

  @Override
  public FromPayload withoutPayload() {
    return this;
  }

  @Override
  public FromPathParameter pathParameter(String... pathParameters) {
    this.pathParameters = Arrays.asList(pathParameters);
    return this;
  }

  @Override
  public FromPathParameter noPathParameter() {
    return this;
  }

  @Override
  public FromQuery query(List<QueryParameter> queryParameters) {
    if (queryParameters != null) {
      this.queryParameters.addAll(queryParameters);
    }
    return this;
  }

  @Override
  public FromQuery noQuery() {
    return this;
  }

  @Override
  public FromRequestEnhancer enhancer(RequestEnhancer requestEnhancer) {
    this.requestEnhancer = requestEnhancer;
    return this;
  }

  @Override
  public FromRequestEnhancer noEnhancer() {
    return this;
  }

  @Override
  public End insertHeaderFields(Map<String, String> headerFields) {
    this.headerFields.putAll(headerFields);
    return this;
  }

  public static FromCreate createBuilder(ServiceEndpoint serviceEndpoint) {
    return new ServiceCallBuilder(serviceEndpoint);
  }

  @Override
  public ServiceCall end() throws InternalException {
    if (serviceEndpoint == null) {
      throw new InternalException("No service endpoint specified");
    }
    if (pathParameters != null) {
      for (var pathParameter : pathParameters) {
        if (StringUtils.isBlank(pathParameter)) {
          throw new InternalException(
              "An empty path parameter was specified for service " + serviceEndpoint.getPath());
        }
      }
    }
    return new ServiceCallImpl(
        httpMethod,
        serviceEndpoint,
        pathParameters,
        payload,
        headerFields,
        queryParameters,
        requestEnhancer);
  }
}
