/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.model;

import com.sap.cds.impl.parser.ExprParser;
import com.sap.cds.ql.BooleanValue;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsAnnotation;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/** Supported CDS annotations */
public enum CdsAnnotations {
  RESTRICT(null, "restrict"),
  REQUIRES(null, "requires"),

  READONLY(false, "readonly"),
  INSERTONLY(false, "insertonly"),

  INSERTABLE(true, "Capabilities.Insertable", "Capabilities.InsertRestrictions.Insertable"),
  UPDATABLE(true, "Capabilities.Updatable", "Capabilities.UpdateRestrictions.Updatable"),
  DELETABLE(true, "Capabilities.Deletable", "Capabilities.DeleteRestrictions.Deletable"),
  READABLE(true, "Capabilities.Readable", "Capabilities.ReadRestrictions.Readable"),
  UPDATABLE_DELTA(false, "Capabilities.UpdateRestrictions.DeltaUpdateSupported"),

  AUTOEXPOSED(
      false,
      "cds.autoexposed"), // indicates that an entity is auto exposed (explicitly or automatically)
  AUTOEXPOSE(false, "cds.autoexpose"), // used to explicitly auto expose an entity

  ON_INSERT(null, "cds.on.insert", "odata.on.insert"),
  ON_UPDATE(null, "cds.on.update", "odata.on.update"),

  INAPPLICABLE(false, "inapplicable", "disabled"),
  MANDATORY(false, "mandatory", "Common.FieldControl.Mandatory", "FieldControl.Mandatory"),
  FIELD_CONTROL_READONLY(false, "Common.FieldControl.ReadOnly", "FieldControl.ReadOnly"),
  COMMON_FIELDCONTROL(null, "Common.FieldControl"),
  ASSERT_RANGE(false, "assert.range"),
  ASSERT_FORMAT(false, "assert.format"),
  ASSERT_NOTNULL(true, "assert.notNull"),
  ASSERT_CONSTRAINT(false, "assert.constraint"),

  ASSERT_TARGET(false, "assert.target"),

  QUERY_LIMIT_DEFAULT(null, "cds.query.limit.default", "cds.query.limit"),
  QUERY_LIMIT_MAX(null, "cds.query.limit.max"),
  DEFAULT_ORDER(null, "cds.default.order", "odata.default.order"),

  CORE_COMPUTED(false, "Core.Computed"),
  CORE_IMMUTABLE(false, "Core.Immutable"),
  CORE_MEDIA_TYPE(null, "Core.MediaType"),
  CORE_CONTENT_DISPOSITION_FILENAME(null, "Core.ContentDisposition.Filename"),
  CORE_CONTENT_DISPOSITION_TYPE(null, "Core.ContentDisposition.Type"),

  SINGLETON(false, "odata.singleton", "odata.singleton.nullable"),

  DRAFT_ANNOTATION(false, "odata.draft.enabled"),
  DRAFT_PREPARE_ANNOTATION(null, "Common.DraftNode.PreparationAction"),
  DRAFT_PERSISTENCE_ANNOTATION(null, "odata.draft.persistence"),

  ODATA_FOREIGN_KEY_FOR(null, "odata.foreignKey4"),

  ODATA_APPLY_TRANSFORMATIONS(false, "odata.apply.transformations"),

  PATH(null, "endpoints.path", "path"),
  PROTOCOLS(null, "endpoints.protocol", "protocols", "protocol"),
  ENDPOINTS(null, "endpoints"),
  IGNORE(false, "cds.ignore"),
  SERVE_IGNORE(false, "cds.serve.ignore"),

  TOPIC(null, "topic"),
  KAFKA_TOPIC(null, "kafka.topic"),

  ETAG(false, "odata.etag"),

  AGGREGATION_DEFAULT(null, "Aggregation.default"),
  SEMANTICS_CURRENCY_CODE(false, "Semantics.currencyCode"),
  SEMANTICS_UNIT_OR_CURRENCY_REF(
      null, "Semantics.amount.currencyCode", "Semantics.quantity.unitOfMeasure"),
  SEMANTICS_UNIT_OF_MEASURE(false, "Semantics.unitOfMeasure"),

  PERSONALDATA_ENTITYSEMANTICS(null, "PersonalData.EntitySemantics"),
  PERSONALDATA_DATASUBJECTROLE(null, "PersonalData.DataSubjectRole"),
  PERSONALDATA_FIELDSEMANTICS(null, "PersonalData.FieldSemantics"),
  PERSONALDATA_POTENTIALLYPERSONAL(null, "PersonalData.IsPotentiallyPersonal"),
  PERSONALDATA_POTENTIALLYSENSITIVE(null, "PersonalData.IsPotentiallySensitive"),
  CASCADE_DELETE(null, "cascade.delete"),

  // Defines internal operation type created for validation purposes
  INTERNAL_OPERATION_TYPE(false, "cds.internal.operationType"),

  COMMON_LABEL("", "Common.Label"),
  TITLE("", "title"),
  CHANGELOG(false, "changelog"),

  // Marks everything that is related to the changelog storage
  CHANGELOG_INTERNAL_STORAGE(false, "changelog.internal.storage"),
  // Used to recognize the elements of the changelog in the projections or inlines
  CHANGELOG_INTERNAL_SEMANTICS("", "changelog.internal.semantics"),
  // Enables backend embellishments of the changelog elements: texts, odata path etc.
  CHANGELOG_INTERNAL_ENRICH(false, "changelog.internal.enrich"),
  COMMON_TEXT(Map.of(), "Common.Text"),

  HTTP_CACHE_CONTROL_MAX_AGE(null, "http.CacheControl.maxAge");

  private final String[] names;
  private final Object defaultValue;

  private CdsAnnotations(Object defaultValue, String... names) {
    this.defaultValue = defaultValue;
    this.names = names; // NOSONAR
  }

  @SuppressWarnings("unchecked")
  public <T> T getOrDefault(CdsAnnotatable cdsModelElement) {
    return (T) getOrValue(cdsModelElement, this.defaultValue);
  }

  public boolean isTrue(CdsAnnotatable cdsModelElement) {
    Object value = getOrDefault(cdsModelElement);
    return Boolean.TRUE.equals(value) || CQL.TRUE.equals(value);
  }

  public boolean isExpression(CdsAnnotatable cdsModelElement) {
    Object value = getOrDefault(cdsModelElement);
    return value instanceof CqnValue && !(value instanceof BooleanValue);
  }

  public CqnPredicate asPredicate(CdsAnnotatable cdsModelElement) {
    return new ExprParser().parsePredicate(((CqnValue) getOrDefault(cdsModelElement)).tokens());
  }

  @SuppressWarnings("unchecked")
  public <T> List<T> getListOrDefault(CdsAnnotatable cdsModelElement) {
    return (List<T>) getListOrValue(cdsModelElement, this.defaultValue);
  }

  @SuppressWarnings("unchecked")
  public <T> List<T> getListOrValue(CdsAnnotatable cdsModelElement, T value) {

    Object annotationValue = getOrValue(cdsModelElement, value);
    if (annotationValue != null) {
      if (annotationValue instanceof List) {
        return (List<T>) annotationValue;
      } else {
        return (List<T>) Arrays.asList(annotationValue);
      }
    }
    return null;
  }

  public <T> T getOrValue(CdsAnnotatable cdsModelElement, T value) {

    for (String name : names) {
      Optional<CdsAnnotation<T>> annotation = cdsModelElement.findAnnotation(name);
      if (annotation.isPresent()) {
        T annotationValue = annotation.get().getValue();
        if (annotationValue != null && !annotationValue.equals(value)) {
          return annotationValue;

        } else if (annotationValue == null && value != null) {
          return annotationValue;
        }
      }
    }

    return value;
  }
}
