/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.model;

import static com.sap.cds.impl.parser.token.CqnBoolLiteral.TRUE;

import com.google.common.collect.Maps;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSource;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.impl.DeleteBuilder;
import com.sap.cds.ql.impl.SelectBuilder;
import com.sap.cds.ql.impl.UpdateBuilder;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.DataUtils;
import java.util.HashMap;
import java.util.Map;

public class CqnUtils {

  private CqnUtils() {
    // hidden
  }

  /**
   * Converts the object array to an indexed map. The first object has the key <code>0</code>, the
   * second <code>1</code>, etc.
   *
   * @param paramValues the objects
   * @return the indexed map
   */
  public static Map<String, Object> convertToIndexMap(Object... paramValues) {
    Map<String, Object> parameters = new HashMap<>();
    for (int i = 0; i < paramValues.length; i++) {
      parameters.put(String.valueOf(i), paramValues[i]);
    }
    return parameters;
  }

  /**
   * Creates a {@link CqnStatement} with a modified where clause according to the passed function.
   * In case of a nested select, the modification applies to inner select only.
   *
   * @param <S> The type of the statement
   * @param cqn The original {@link CqnStatement}.
   * @param pred a predicate, which is set as the where clause or connected to an existing where
   *     clause with and
   * @return The modified {@link CqnStatement}.
   */
  @SuppressWarnings("unchecked")
  public static <S extends CqnStatement> S addWhere(final S cqn, CqnPredicate pred) {
    if (pred == null || pred == TRUE) {
      return cqn;
    }
    if (cqn.isSelect()) {
      SelectBuilder<?> s = (SelectBuilder<?>) cqn;
      return (S) SelectBuilder.copyShallow(s).filterSource(pred);
    } else if (cqn.isUpdate()) {
      UpdateBuilder<?> u = (UpdateBuilder<?>) cqn;
      return (S) UpdateBuilder.copyShallow(u).filter(pred);
    } else if (cqn.isDelete()) {
      DeleteBuilder<?> d = (DeleteBuilder<?>) cqn;
      return (S) DeleteBuilder.copyShallow(d).filter(pred);
    } else {
      throw new IllegalStateException(
          "Unexpected statement type: " + cqn.getClass().getCanonicalName());
    }
  }

  /**
   * Transform the {@link CqnSelect} into a {@link Delete}
   *
   * @param s the {@link CqnSelect}
   * @return the {@link Delete}
   */
  public static Delete<?> toDelete(CqnSelect s) {
    Delete<StructuredType<?>> delete = Delete.from(s.ref());
    s.where().ifPresent(delete::where);
    return delete;
  }

  /**
   * Transforms a {@link CqnDelete} into a {@link Select}
   *
   * @param delete the {@link CqnDelete}
   * @return the {@link Select}
   */
  public static Select<?> toSelect(CqnDelete delete) {
    Select<?> select = Select.from(delete.ref());
    delete.where().ifPresent(select::where);
    return select;
  }

  /**
   * Transforms a {@link CqnUpdate} into a {@link Select}
   *
   * @param update the {@link CqnUpdate}
   * @return the {@link Select}
   */
  public static Select<?> toSelect(CqnUpdate update, CdsEntity target) {
    Select<?> select = Select.from(update.ref());
    update.where().ifPresent(select::where);
    CqnPredicate dataCondition =
        update.entries().stream()
            .map(entry -> DataUtils.keyValues(target, entry))
            .map(keys -> Maps.filterValues(keys, v -> v != null))
            .map(CQL::matching)
            .collect(CQL.withOr());
    return addWhere(select, dataCondition);
  }

  public static CqnStructuredTypeRef getTargetRef(
      CqnStatement statement, boolean allowTableFunctions) {
    if (statement.isSelect()) {
      return getTargetRef(statement.asSelect().from(), allowTableFunctions);
    }
    return statement.ref();
  }

  private static CqnStructuredTypeRef getTargetRef(CqnSource source, boolean allowTableFunctions) {
    if (source.isSelect()) {
      return getTargetRef(source.asSelect().from(), allowTableFunctions);
    }
    if (source.isTableFunction()) {
      if (!allowTableFunctions) {
        throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_TABLE_FUNCTION);
      }
      return getTargetRef(source.asTableFunction().source(), allowTableFunctions);
    }
    return source.asRef();
  }
}
