/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.model;

import com.sap.cds.services.environment.CdsProperties.Model.Provider;
import com.sap.cds.services.environment.CdsProperties.Model.Provider.Cache;
import com.sap.cds.services.environment.CdsProperties.MultiTenancy.Sidecar;
import com.sap.cds.services.mt.ExtensibilityService;
import com.sap.cds.services.mt.ModelChangedEventContext;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.IdentityUtils;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.lib.mtx.MetaDataAccessor;
import com.sap.cds.services.utils.lib.mtx.ModelId;
import com.sap.cds.services.utils.lib.mtx.impl.CacheParams;
import com.sap.cds.services.utils.lib.mtx.impl.MetaDataAccessorImpl;
import com.sap.cds.services.utils.lib.mtx.impl.MetaDataAccessorImpl.CdsModelCreator;
import com.sap.cds.services.utils.lib.mtx.impl.MetaDataAccessorImpl.EdmxModelCreator;
import com.sap.cds.services.utils.lib.mtx.impl.MetaDataAccessorImpl.I18nResourceCreator;
import com.sap.cds.services.utils.lib.mtx.impl.ModelProviderAccess;
import com.sap.cds.services.utils.lib.mtx.impl.SidecarAccess;
import com.sap.cds.services.utils.lib.tools.api.ResilienceConfig;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.sdk.cloudplatform.connectivity.BtpServiceOptions;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultHttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.OnBehalfOf;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationOptions;
import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DynamicModelUtils {
  private static final Logger logger = LoggerFactory.getLogger(DynamicModelUtils.class);

  private final CdsRuntime runtime;
  private final Provider providerConfig;
  private final Sidecar sidecarConfig;
  private final IdentityUtils identityUtils;

  public DynamicModelUtils(CdsRuntime runtime) {
    this.runtime = runtime;
    this.providerConfig = runtime.getEnvironment().getCdsProperties().getModel().getProvider();
    this.sidecarConfig = runtime.getEnvironment().getCdsProperties().getMultiTenancy().getSidecar();
    this.identityUtils = new IdentityUtils(runtime);
  }

  /**
   * Returns true, if the static model, packaged with the application, can be loaded Retrieves the
   * required information from the current {@link RequestContext}
   *
   * @return true, if the static model, packaged with the application, can be loaded
   */
  public boolean useStaticModel() {
    RequestContext requestContext = RequestContext.getCurrent(runtime);
    return useStaticModel(requestContext.getUserInfo(), requestContext.getFeatureTogglesInfo());
  }

  /**
   * Returns true, if the static model, packaged with the application, can be loaded
   *
   * @param userInfo the {@link UserInfo}
   * @param featureTogglesInfo the {@link FeatureTogglesInfo}
   * @return true, if the static model, packaged with the application, can be loaded
   */
  public boolean useStaticModel(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
    boolean hasAllToggles =
        featureTogglesInfo.getEnabledFeatureToggles().anyMatch(t -> t.getName().trim().equals("*"));
    return (!providerConfig.isExtensibility() || userInfo.getTenant() == null)
        && (!providerConfig.isToggles() || hasAllToggles);
  }

  /**
   * Prepares the {@link ModelId} with tenant and feature toggles dimension, depending on the
   * configuration
   *
   * @param userInfo the {@link UserInfo}
   * @param featureTogglesInfo the {@link FeatureTogglesInfo}
   * @return the {@link ModelId} with tenant and feature toggles dimension, depending on the
   *     configuration
   */
  public ModelId.Builder prepareModelId(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
    String tenant = providerConfig.isExtensibility() ? userInfo.getTenant() : null;
    Set<String> features =
        providerConfig.isToggles()
            ? featureTogglesInfo
                .getEnabledFeatureToggles()
                .map(ft -> ft.getName())
                .collect(Collectors.toSet())
            : Collections.emptySet();
    return ModelId.create(tenant).features(features);
  }

  /**
   * @return {@code true} if ModelProviderService is configured
   */
  public boolean isModelProviderEnabled() {
    return !StringUtils.isEmpty(getModelProviderUrl())
        && (providerConfig.isExtensibility() || providerConfig.isToggles());
  }

  public String getModelProviderUrl() {
    return !StringUtils.isEmpty(providerConfig.getUrl())
        ? providerConfig.getUrl()
        : sidecarConfig.getUrl();
  }

  public <T> MetaDataAccessor<T> createMetadataAccessor(
      EdmxModelCreator<T> strToEdmx, CdsModelCreator strToModel, I18nResourceCreator strToI18n) {
    // create accessor
    SidecarAccess sidecarAccess = new ModelProviderAccess(request -> {}, getResilienceConfig());
    CacheParams cacheParams = getCacheParams(providerConfig.getCache());
    MetaDataAccessor<T> accessor =
        new MetaDataAccessorImpl<>(
            new MetaDataAccessorImpl.MetaDataAccessorConfig.Builder()
                .sidecarAccess(sidecarAccess)
                .cacheParams(cacheParams)
                .strToEdmx(strToEdmx)
                .strToModel(strToModel)
                .strToI18n(strToI18n)
                .build(),
            null);

    // register accessor with extensibility service
    if (accessor != null) {
      ExtensibilityService extService =
          runtime
              .getServiceCatalog()
              .getService(ExtensibilityService.class, ExtensibilityService.DEFAULT_NAME);
      if (extService != null) {
        extService.on(
            ExtensibilityService.EVENT_MODEL_CHANGED,
            null,
            context -> {
              String tenant = context.getUserInfo().getTenant();
              Instant changedAt = context.as(ModelChangedEventContext.class).getTimestamp();
              // would be better to pass the timestamp on here instead of maxAge
              // time diff between this maxAge calculation and calculation of actualAge can lead to
              // unnecessary refreshes
              int maxAge =
                  changedAt == null
                      ? 0
                      : (int) Duration.between(changedAt, Instant.now()).toSeconds();
              accessor.refresh(tenant, maxAge);
            });
      }
    }

    return accessor;
  }

  private CacheParams getCacheParams(Cache cache) {
    long maximumSize = cache.getMaxSize();
    long expirationDuration = cache.getExpirationTime();
    long refreshDuration = cache.getRefreshTime();
    return new CacheParams(
        maximumSize,
        Duration.ofSeconds(expirationDuration),
        Duration.ofSeconds(refreshDuration),
        false,
        false);
  }

  public ResilienceConfig getResilienceConfig() {
    return ResilienceConfig.builder()
        .numOfRetries(3)
        .baseWaitTime(Duration.ofMillis(500))
        .waitTimeCalculation(ResilienceConfig.WaitTimeCalculation.LINEAR)
        .build();
  }

  public HttpDestination createSidecarDestination(String destinationName, String targetUrl) {
    List<ServiceBinding> uaaBindings = identityUtils.getXsuaaServiceBindings();
    ServiceBinding binding;

    if (!uaaBindings.isEmpty()) {
      binding = uaaBindings.get(0);
    } else if (!identityUtils.getIasServiceBindings().isEmpty()) {
      binding = identityUtils.getIasServiceBindings().get(0);
    } else {
      logger.debug(
          "Initializing MTX sidecar destination '{}' without service binding.", destinationName);
      return DefaultHttpDestination.builder(targetUrl).name(destinationName).build();
    }

    logger.debug(
        "Initializing MTX sidecar destination '{}' using service binding '{}'.",
        destinationName,
        binding.getName().get());

    DefaultHttpDestination http =
        (DefaultHttpDestination)
            ServiceBindingDestinationLoader.defaultLoaderChain()
                .getDestination(
                    ServiceBindingDestinationOptions.forService(binding)
                        .withOption(
                            BtpServiceOptions.AuthenticationServiceOptions.withTargetUri(targetUrl))
                        .onBehalfOf(OnBehalfOf.TECHNICAL_USER_PROVIDER)
                        .build());

    return DefaultHttpDestination.fromDestination(http).name(destinationName).build();
  }
}
