/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.outbox;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.outbox.OutboxMessage;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.outbox.StoredRequestContext;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.RequestContextRunner;
import com.sap.cds.services.utils.StringUtils;
import java.time.Instant;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

public class OutboxUtils {

  public static final String OUTBOX_MODEL = "cds.outbox.Messages";

  /**
   * Determines whether the outbox model is available.
   *
   * @param model CDS model
   * @return <code>true</code> if outbox model is available and <code>false</code> otherwise.
   */
  public static boolean hasOutboxModel(CdsModel model) {
    return model.findEntity(OUTBOX_MODEL).isPresent();
  }

  /**
   * Wraps the given service with the outbox. If the outbox is not available the in-memory outbox is
   * used instead if available. If no outbox is available at all the original service instance is
   * returned.
   *
   * @param <S> the service type
   * @param service the service to outbox
   * @param outboxName the name of the preferred outbox
   * @param runtime the {@link CdsRuntime}
   * @return the outbox-wrapped service, if a outbox was available
   */
  public static <S extends Service> S outboxed(S service, String outboxName, CdsRuntime runtime) {
    OutboxService outbox = null;
    if (!StringUtils.isEmpty(outboxName)) {
      outbox = runtime.getServiceCatalog().getService(OutboxService.class, outboxName);
    }
    if (outbox == null) {
      outbox =
          runtime.getServiceCatalog().getService(OutboxService.class, OutboxService.INMEMORY_NAME);
    }
    if (outbox != null) {
      return outbox.outboxed(service);
    }
    return service;
  }

  /**
   * Turns the {@link EventContext} generically into a {@link OutboxMessage}
   *
   * @param context the {@link EventContext}
   * @return the {@link OutboxMessage}
   */
  public static OutboxMessage toOutboxMessage(EventContext context) {
    Map<String, Object> contextParams = new HashMap<>();
    context.keySet().forEach(k -> contextParams.put(k, context.get(k)));
    OutboxMessage message = OutboxMessage.create();
    if (context.getTarget() != null) {
      message.setEntity(context.getTarget().getQualifiedName());
    }
    message.setEvent(context.getEvent());
    message.setParams(contextParams);
    return message;
  }

  /**
   * Turns a {@link OutboxMessageEventContext} and its {@link OutboxMessage} generically into an
   * {@link EventContext}
   *
   * @param outboxContext the {@link OutboxMessageEventContext}
   * @return the {@link EventContext}
   */
  public static EventContext toEventContext(OutboxMessageEventContext outboxContext) {
    OutboxMessage message = outboxContext.getMessage();
    EventContext context = EventContext.create(message.getEvent(), message.getEntity());
    if (message.getParams() != null) {
      message.getParams().forEach(context::put);
    }
    return context;
  }

  public static void storeRequestContext(RequestContext requestContext, OutboxMessage message) {
    if (message.getStoredRequestContext() == null) {
      message.setStoredRequestContext(StoredRequestContext.create());
    }
    StoredRequestContext stored = message.getStoredRequestContext();

    UserInfo userInfo = requestContext.getUserInfo();
    stored.setTenant(userInfo.getTenant()); // overwrite tenant always
    if (stored.getIsPrivileged() == null) {
      stored.setIsPrivileged(userInfo.isPrivileged());
    }

    ParameterInfo parameterInfo = requestContext.getParameterInfo();
    Locale locale = parameterInfo.getLocale();
    Instant validFrom = parameterInfo.getValidFrom();
    Instant validTo = parameterInfo.getValidTo();
    // nullable
    stored.putIfAbsent("correlationId", parameterInfo.getCorrelationId());
    stored.putIfAbsent("locale", locale != null ? locale.toLanguageTag() : null);
    stored.putIfAbsent("validFrom", validFrom != null ? validFrom.toString() : null);
    stored.putIfAbsent("validTo", validTo != null ? validTo.toString() : null);
  }

  public static void restoreRequestContext(
      RequestContextRunner requestContextRunner, OutboxMessage message) {
    StoredRequestContext stored = message.getStoredRequestContext();
    if (stored != null) {
      requestContextRunner
          .modifyParameters(
              paramInfo -> {
                String locale = stored.getLocale();
                String validFrom = stored.getValidFrom();
                String validTo = stored.getValidTo();
                paramInfo.setCorrelationId(stored.getCorrelationId());
                paramInfo.setLocale(locale != null ? Locale.forLanguageTag(locale) : null);
                paramInfo.setValidFrom(validFrom != null ? Instant.parse(validFrom) : null);
                paramInfo.setValidTo(validTo != null ? Instant.parse(validTo) : null);
              })
          .systemUser(stored.getTenant());
      if (stored.getIsPrivileged()) {
        requestContextRunner.privilegedUser();
      }
    }
  }
}
