/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.path;

import com.sap.cds.reflect.CdsDefinition;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/** A builder for creating {@link CdsResourcePath}s conveniently. */
public class CdsResourcePathBuilder {

  /** Internal implementation */
  private static class CdsResourcePathImpl implements CdsResourcePath {

    protected String path;

    protected boolean isPublic;

    protected final CdsDefinition cdsDefinition;

    protected Set<String> publicHttpMethods = new HashSet<>();

    protected List<CdsResourcePath> subPaths = new ArrayList<>();

    protected CdsResourcePathImpl(CdsDefinition cdsDefinition) {
      this.cdsDefinition = cdsDefinition;
      this.path = cdsDefinition.getName();
    }

    protected void setPaths(String... paths) {
      this.path = append(paths);
    }

    protected void consolidate() {
      // a public endpoint implicitly has open http events
      if (isPublic) {
        publicHttpMethods.clear();
      }
    }

    @Override
    public String getPath() {
      return this.path;
    }

    @Override
    public boolean isPublic() {
      return this.isPublic;
    }

    @Override
    public Stream<String> publicEvents() {
      return publicHttpMethods.stream();
    }

    @Override
    public Stream<CdsResourcePath> subPaths() {
      return subPaths.stream();
    }

    @Override
    public CdsDefinition getCdsDefinition() {
      return cdsDefinition;
    }

    private static String normalize(String path) {
      String[] pathParts = path.replaceAll("\\s+", "").split("\\.");
      return Stream.of(pathParts)
          .filter(part -> part != null && !part.isEmpty())
          .collect(Collectors.joining("."));
    }

    private static String append(String... pathParts) {
      return Stream.of(pathParts)
          .map(CdsResourcePathImpl::normalize)
          .collect(Collectors.joining("."));
    }
  }

  /** The ResourcePath being constructed */
  private CdsResourcePathImpl cdsResourcePath;

  public static CdsResourcePathBuilder cds(CdsDefinition cdsDefinition) {
    CdsResourcePathBuilder builder = new CdsResourcePathBuilder();
    builder.cdsResourcePath = new CdsResourcePathImpl(cdsDefinition);
    return builder;
  }

  /**
   * Sets the path of the resource.
   *
   * @param paths The path parts
   * @return The builder
   */
  public CdsResourcePathBuilder path(String... paths) {
    cdsResourcePath.setPaths(paths);
    return this;
  }

  /**
   * Specifies if the current path should be public accessible
   *
   * @param isPublic if {@code true} the path should be accessible for public
   * @return The builder
   */
  public CdsResourcePathBuilder isPublic(boolean isPublic) {
    cdsResourcePath.isPublic = isPublic;
    return this;
  }

  /**
   * Specifies public events. Makes only sense for non-public endpoints.
   *
   * @param publicEvents The public events
   * @return The builder
   */
  public CdsResourcePathBuilder publicEvents(Stream<String> publicEvents) {
    publicEvents.collect(Collectors.toCollection(() -> cdsResourcePath.publicHttpMethods));
    return this;
  }

  /**
   * Adds a new {@code CdsResourcePath} as child of the current path.
   *
   * @param subPath The sub path to be added
   * @return The builder
   */
  public CdsResourcePathBuilder subPath(CdsResourcePath subPath) {
    if (!cdsResourcePath.subPaths.stream().anyMatch(sp -> sp.getPath().equals(subPath.getPath()))) {
      cdsResourcePath.subPaths.add(subPath);
    }
    return this;
  }

  /**
   * Adds new {@code CdsResourcePath}s as children of the current path.
   *
   * @param subPaths The sub paths to be added
   * @return The builder
   */
  public CdsResourcePathBuilder subPaths(Stream<CdsResourcePath> subPaths) {
    subPaths.forEach(subPath -> subPath(subPath));
    return this;
  }

  /**
   * Finally constructs the {@link CdsResourcePath}
   *
   * @return The constructed {@link CdsResourcePath}
   */
  public CdsResourcePath build() {
    cdsResourcePath.consolidate();
    return cdsResourcePath;
  }
}
