/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.path;

import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.utils.StringUtils;

public class UrlPathUtil {

  public static enum HttpMethod {
    GET,
    POST,
    HEAD,
    OPTIONS,
    PUT,
    PATCH,
    DELETE,
    TRACE;
  }

  /**
   * Maps a CDS event such as "READ" or "CREATE" to corresponding HTTP methods"GET" resp. "POST".
   * The mapping is inspired by OData protocol handling. The matching is case-sensitive.
   *
   * @param cdsEvent The CDS event
   * @return The HTTP event or {@code null} if no matching HTTP event available.
   */
  public static String cdsEvent2HttpMethod(String cdsEvent) {

    switch (cdsEvent) {
      case (CqnService.EVENT_READ):
        return HttpMethod.GET.name();

      case (CqnService.EVENT_DELETE):
        return HttpMethod.DELETE.name();

      case (CqnService.EVENT_CREATE):
        return HttpMethod.POST.name();

      case (CqnService.EVENT_UPSERT):
        return HttpMethod.PUT.name();

      case (CqnService.EVENT_UPDATE):
        return HttpMethod.PATCH.name();

      default:
        return null;
    }
  }

  /**
   * Normalizes a base path of a servlet-based adapter
   *
   * @param basePath the configured base path
   * @return the normalized base path
   */
  public static String normalizeBasePath(String basePath) {
    return UrlResourcePathBuilder.path("/" + StringUtils.trim(basePath, '/')).build().getPath();
  }
}
