/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.path;

import com.sap.cds.adapter.UrlResourcePath;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/** A builder for creating {@link UrlResourcePath}s conveniently. */
public class UrlResourcePathBuilder {

  /** Internal implementation */
  private static class UrlResourcePathImpl implements UrlResourcePath {

    protected String path;

    protected boolean isPublic;

    protected boolean recursive;

    protected boolean directory;

    protected Set<String> publicHttpMethods = new HashSet<>();

    protected List<UrlResourcePath> subPaths = new ArrayList<>();

    private static final String RECURSIVE_PATH = "/**";

    private static final String DIRECTORY_PATH = "/*";

    private static String normalize(String path) {
      String[] pathParts = path.replaceAll("\\s+", "").split("/");
      return "/"
          + Stream.of(pathParts)
              .filter(part -> part != null && !part.isEmpty())
              .collect(Collectors.joining("/"));
    }

    private static String append(String... pathParts) {
      if (pathParts.length > 0) {
        String result =
            Stream.of(pathParts)
                .map(UrlResourcePathImpl::normalize)
                .filter(p -> !p.equals("/"))
                .collect(Collectors.joining());
        if (result.isEmpty()
            || (pathParts.length > 1 && pathParts[pathParts.length - 1].endsWith("/"))) {
          result += "/";
        }
        return result;
      }
      return null;
    }

    private static String recursive(String path) {
      String normalizedPath = normalize(path);
      if (!normalizedPath.endsWith(RECURSIVE_PATH)) {
        return append(normalizedPath, RECURSIVE_PATH);
      }
      return normalizedPath;
    }

    private static String directory(String path) {
      String normalizedPath = normalize(path);
      if (!normalizedPath.endsWith(DIRECTORY_PATH)) {
        return normalizedPath + DIRECTORY_PATH;
      }
      return normalizedPath;
    }

    protected void setPaths(String... paths) {
      this.path = append(paths);
    }

    protected void consolidate() {

      // a public endpoint implicitly has open http events
      if (isPublic) {
        publicHttpMethods.clear();
      }

      // adapt the sub path with the base path
      forAllSubPathsRecursively(
          this, p -> ((UrlResourcePathImpl) p).path = append(path, p.getPath()));

      // finally adapt the path itself
      if (recursive) {
        path = recursive(path);
      } else if (directory) {
        path = directory(path);
      }
    }

    private void forAllSubPathsRecursively(UrlResourcePath path, Consumer<UrlResourcePath> func) {
      path.subPaths()
          .forEach(
              p -> {
                func.accept(p);
                forAllSubPathsRecursively(p, func);
              });
    }

    @Override
    public String getPath() {
      return this.path;
    }

    @Override
    public boolean isPublic() {
      return this.isPublic;
    }

    @Override
    public Stream<String> publicEvents() {
      return publicHttpMethods.stream();
    }

    @Override
    public Stream<UrlResourcePath> subPaths() {
      return subPaths.stream();
    }
  }

  /** The ResourcePath being constructed */
  private UrlResourcePathImpl servicePath;

  /**
   * Sets the path of the resource.
   *
   * @param paths The path parts
   * @return The builder
   */
  public static UrlResourcePathBuilder path(String... paths) {

    UrlResourcePathBuilder builder = new UrlResourcePathBuilder();
    builder.servicePath = new UrlResourcePathImpl();
    builder.servicePath.setPaths(paths);
    return builder;
  }

  /**
   * Specifies if the current path should be public accessible
   *
   * @param isPublic if {@code true} the path should be accessible for public
   * @return The builder
   */
  public UrlResourcePathBuilder isPublic(boolean isPublic) {
    servicePath.isPublic = isPublic;
    return this;
  }

  /**
   * Specifies that the current path includes all sub paths
   *
   * @return The builder
   */
  public UrlResourcePathBuilder recursive() {
    servicePath.recursive = true;
    return this;
  }

  /**
   * Specifies that the current path includes all endpoints in the directory
   *
   * @return The builder
   */
  public UrlResourcePathBuilder directory() {
    servicePath.directory = true;
    return this;
  }

  /**
   * Specifies public events. Makes only sense for non-public endpoints.
   *
   * @param publicEvents The public events
   * @return The builder
   */
  public UrlResourcePathBuilder publicEvents(Stream<String> publicEvents) {
    publicEvents.collect(Collectors.toCollection(() -> servicePath.publicHttpMethods));
    return this;
  }

  /**
   * Adds a new {@code ResourcePath} as child of the current path.
   *
   * @param subPath The sub path to be added
   * @return The builder
   */
  public UrlResourcePathBuilder subPath(UrlResourcePath subPath) {
    if (!servicePath.subPaths.stream().anyMatch(sp -> sp.getPath().equals(subPath.getPath()))) {
      servicePath.subPaths.add(subPath);
    }
    return this;
  }

  /**
   * Adds new {@code ResourcePath}s as children of the current path.
   *
   * @param subPaths The sub paths to be added
   * @return The builder
   */
  public UrlResourcePathBuilder subPaths(Stream<UrlResourcePath> subPaths) {
    subPaths.forEach(subPath -> subPath(subPath));
    return this;
  }

  /**
   * Finally constructs the ResourcePath
   *
   * @return The constructed ResourcePath
   */
  public UrlResourcePath build() {
    servicePath.consolidate();
    return servicePath;
  }
}
