/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils;

import com.sap.cds.services.request.ParameterInfo;
import java.util.UUID;
import java.util.function.UnaryOperator;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;

/** Utility for retrieving or generating a correlation id. */
public class CorrelationIdUtils {

  public static final String CORRELATION_ID_HEADER_KEY = "X-CorrelationID";

  public static final String CORRELATION_ID_HEADER_KEY_ALIAS = "X-Correlation-Id";

  public static final String CORRELATION_ID_FIELD = "correlation_id";

  public static final String CORRELATION_ID_UNKNOWN =
      "-"; // used by cf-java-logging-support as "unknown" value

  private CorrelationIdUtils() {}

  /**
   * Checks whether there's a correlation id entry in the {@link MDC}.
   *
   * @return true, if there's an entry under the key "correlation_id"
   */
  public static boolean mdcHasEntry() {
    return !isUnknown(getFromMDC());
  }

  /**
   * Puts the provided correlation id into the {@link MDC} under the key "correlation_id".
   *
   * @param correlationId the correlation id
   */
  public static void putInMDC(String correlationId) {
    if (!isUnknown(correlationId)) {
      MDC.put(CORRELATION_ID_FIELD, correlationId);
    }
  }

  /**
   * Gets a correlation id value from the MDC, if existing, or null otherwise.
   *
   * @return a correlation id
   */
  public static String getFromMDC() {
    return MDC.get(CorrelationIdUtils.CORRELATION_ID_FIELD);
  }

  /** Removes the correlation id field from the {@link MDC}. */
  public static void clearMDC() {
    MDC.remove(CORRELATION_ID_FIELD);
  }

  /**
   * Returns the correlation id if already existing, or generates a new one otherwise.
   *
   * <p>Tries to retrieve an existing correlation id from a MDC field, or the parameter info
   * headers, in that specific order.
   *
   * @param parameterInfo parameter info to fall back to if correlation id does not exist in MDC
   * @return a correlation id
   */
  public static String getOrGenerateCorrelationId(ParameterInfo parameterInfo) {
    return getCorrelationId(
        parameterInfo,
        correlationId -> isUnknown(correlationId) ? UUID.randomUUID().toString() : correlationId);
  }

  /**
   * Returns the correlation id if already existing, or null if not.
   *
   * @param parameterInfo parameter info to fall back to if correlation id does not exist in MDC,
   *     nullable
   * @return a correlation id
   */
  public static String getCorrelationId(ParameterInfo parameterInfo) {
    return getCorrelationId(parameterInfo, UnaryOperator.identity());
  }

  private static String getCorrelationId(
      ParameterInfo parameterInfo, UnaryOperator<String> correlationIdProvider) {
    String correlationId = getFromMDC();
    if (isUnknown(correlationId) && parameterInfo != null) {
      correlationId = parameterInfo.getHeader(CORRELATION_ID_HEADER_KEY);

      if (isUnknown(correlationId)) {
        correlationId = parameterInfo.getHeader(CORRELATION_ID_HEADER_KEY_ALIAS);
      }
    }
    return correlationIdProvider.apply(correlationId);
  }

  private static boolean isUnknown(String correlationId) {
    return StringUtils.isEmpty(correlationId) || CORRELATION_ID_UNKNOWN.equals(correlationId);
  }
}
