/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils;

import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.model.CdsAnnotations;
import java.util.Iterator;
import java.util.Map;

/** Utility class for the draft handling. */
public class DraftUtils {

  private DraftUtils() {}

  /**
   * Checks if the service has a draft enabled entity.
   *
   * @param serviceName the service name
   * @param model the model
   * @return {@code true} if the service has a draft enabled entity
   */
  public static boolean isDraftEnabled(String serviceName, CdsModel model) {
    return model
        .findService(serviceName)
        .map(s -> s.entities().anyMatch(e -> isDraftEnabled(e)))
        .orElse(false);
  }

  /**
   * Checks is the ref points to a draft instance.
   *
   * @param ref the entity ref
   * @param entity the entity type
   * @param model the model
   * @return true, if the ref points to a draft instance
   */
  public static boolean isDraftTarget(CqnStructuredTypeRef ref, CdsEntity entity, CdsModel model) {
    if (DraftUtils.isDraftEnabled(entity)) {
      Iterator<ResolvedSegment> segments = CqnAnalyzer.create(model).analyze(ref).reverse();
      while (segments.hasNext()) {
        Map<String, Object> targetKeys = segments.next().keyValues();
        if (!targetKeys.isEmpty()) {
          return Boolean.FALSE.equals(targetKeys.get(Drafts.IS_ACTIVE_ENTITY));
        }
      }
    }
    return false;
  }

  /**
   * Checks if the entity is draft enabled.
   *
   * @param entity the entity
   * @return {@code true} if the entity is draft enabled
   */
  public static boolean isDraftEnabled(CdsAnnotatable entity) {
    return CdsAnnotations.DRAFT_ANNOTATION.isTrue(entity)
        || CdsAnnotations.DRAFT_PREPARE_ANNOTATION.getOrDefault(entity) != null;
  }

  public static boolean isDraftRoot(CdsAnnotatable entity) {
    return CdsAnnotations.DRAFT_ANNOTATION.isTrue(entity);
  }
}
