/*
 * © 2023-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.environment;

import com.sap.cloud.environment.servicebinding.api.ServiceBinding;

public class ServiceBindingUtils {

  private ServiceBindingUtils() {
    // avoid instances
  }

  /**
   * Returns true, if a service binding matches the given tag filter or service filter.
   *
   * <p>The tag filter is used first to match the service binding. It is only used if the given tag
   * filter is not {@code null} and if the service binding has a non-empty tag list ({@link
   * com.sap.cloud.environment.servicebinding.api.ServiceBinding#getTags()}). If the tag filter is
   * used and no tag match can be found the serviceFilter is applied.
   *
   * <p>If the tag filter is not used or has a negative result an additional service filter is
   * applied against the service name of the binding. It is only applied if the given service filter
   * is not {@code null}.
   *
   * @param binding the {@link ServiceBinding} to check
   * @param tagFilter the tag to match the tags against.
   * @param serviceNameFilter the service name to match the service against.
   * @return true, if the service binding matches, false otherwise
   */
  public static boolean matches(
      ServiceBinding binding, String tagFilter, String serviceNameFilter) {
    boolean tagsMatched = false;
    if (tagFilter != null && binding.getTags() != null && !binding.getTags().isEmpty()) {
      tagsMatched = binding.getTags().contains(tagFilter);
    }
    return tagsMatched
        || (serviceNameFilter != null
            && serviceNameFilter.equals(binding.getServiceName().orElse(null)));
  }

  /**
   * Returns true, if a service binding matches the given filter. The filter is used as tag filter
   * and service filter.
   *
   * @param binding the {@link ServiceBinding} to check
   * @param filter the filter to match tags and service against
   * @return true, if the service binding matches, false otherwise
   */
  public static boolean matches(ServiceBinding binding, String filter) {
    return matches(binding, filter, filter);
  }
}
