/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.lib.tools.impl;

import static org.apache.http.HttpHeaders.CONTENT_TYPE;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.services.utils.lib.tools.api.HttpMethod;
import com.sap.cds.services.utils.lib.tools.api.QueryParameter;
import com.sap.cds.services.utils.lib.tools.api.ResponseChecker;
import com.sap.cds.services.utils.lib.tools.api.ServiceCall;
import com.sap.cds.services.utils.lib.tools.api.ServiceEndpoint;
import com.sap.cds.services.utils.lib.tools.api.ServiceResponse;
import com.sap.cds.services.utils.lib.tools.exception.ServiceException;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServiceCallImpl implements ServiceCall {
  private static final Logger logger = LoggerFactory.getLogger(ServiceCallImpl.class);
  private static final String APPLICATION_JSON = "application/json";
  private static final String SLASH = "/";
  public static final String RETRY_AFTER = "Retry-After";
  public static final String X_RATE_LIMIT_RETRY_AFTER = "X-RateLimit-Retry-After";
  private final ServiceEndpoint serviceEndpoint;
  private final List<String> pathParameter;
  private final Map<String, String> headerFields = new HashMap<>();
  private final List<QueryParameter> queryParameters = new ArrayList<>();
  private final Object payload;
  private final HttpMethod httpMethod;
  private final ObjectMapper objectMapper;

  protected ServiceCallImpl(
      HttpMethod httpMethod,
      ServiceEndpoint serviceEndpoint,
      List<String> pathParameter,
      Object payload,
      Map<String, String> headerFields,
      List<QueryParameter> queryParameters) {
    this.httpMethod = httpMethod;
    this.serviceEndpoint = serviceEndpoint;
    this.pathParameter = pathParameter;
    this.payload = payload;
    if (headerFields != null) {
      this.headerFields.putAll(headerFields);
    }
    if (queryParameters != null) {
      this.queryParameters.addAll(queryParameters);
    }
    this.objectMapper = new ObjectMapper();
    this.objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
  }

  @Override
  public <R> ServiceResponse<R> execute(Class<R> responseType) throws ServiceException {
    return execute(httpMethod, responseType);
  }

  @Override
  public ServiceResponse<Void> execute() throws ServiceException {
    return execute(httpMethod, Void.class);
  }

  @SuppressWarnings("unchecked")
  private <R> ServiceResponse<R> execute(HttpMethod method, Class<R> responseType)
      throws ServiceException {
    RequestBuilder requestBuilder =
        getRequestBuilder(method, serviceEndpoint.getPath(), pathParameter)
            .setHeader(CONTENT_TYPE, APPLICATION_JSON);
    queryParameters.forEach(
        queryParameter ->
            requestBuilder.addParameter(queryParameter.getName(), queryParameter.getValue()));
    if (payload != null) {
      requestBuilder.setEntity(asEntity(payload));
    }
    HttpUriRequest request = requestBuilder.build();
    headerFields.forEach(request::setHeader);
    Retry retryServiceCall =
        Retry.RetryBuilder.create()
            .retryExceptions(RetryException.class)
            .numOfRetries(serviceEndpoint.getResilienceConfig().getNumOfRetries())
            .baseWaitTime(serviceEndpoint.getResilienceConfig().getBaseWaitTime())
            .waitTimeFunction(serviceEndpoint.getResilienceConfig().getWaitTimeFunction())
            .build();
    try {
      return retryServiceCall.execute(
          () -> execute(request, responseType, serviceEndpoint.getResponseChecker()));
    } catch (Exception exception) {
      RetryException retryException = determineException(exception, RetryException.class);
      NoRetryException noRetryException = determineException(exception, NoRetryException.class);
      if (retryException != null) {
        if (retryException.getCause() == null) {
          // checker didn't throw an exception, user only requested a retry but not an exception =>
          // normal exit
          return (ServiceResponse<R>) retryException.getResponse();
        } else {
          throw new ServiceException(retryException.getCause(), retryException.getResponse());
        }
      } else if (noRetryException != null) {
        throw new ServiceException(noRetryException.getCause(), noRetryException.getResponse());
      }
      // should never happen
      throw new RuntimeException("Wrong exception type");
    }
  }

  private RequestBuilder getRequestBuilder(
      HttpMethod method, String path, List<String> pathParameters) throws ServiceException {
    String totalPath;
    if (path.contains("%s")) {
      totalPath = constructPathFromTemplate(path, pathParameters);
    } else {
      if (pathParameters == null || pathParameters.isEmpty()) {
        totalPath = path;
      } else if (pathParameters.size() > 1) {
        throw new IllegalArgumentException("Too many path parameters specified");
      } else {
        totalPath = addPathParameter(path, pathParameters.get(0));
      }
    }
    switch (method) {
      case GET:
        {
          return RequestBuilder.get(totalPath);
        }
      case PUT:
        {
          return RequestBuilder.put(totalPath);
        }
      case POST:
        {
          return RequestBuilder.post(totalPath);
        }
      case DELETE:
        {
          return RequestBuilder.delete(totalPath);
        }
      case PATCH:
        {
          return RequestBuilder.patch(totalPath);
        }
      default:
        {
          throw new ServiceException("Http method " + method.name() + " not known", null);
        }
    }
  }

  private String constructPathFromTemplate(String path, List<String> pathParameters) {
    int countMatches = StringUtils.countMatches(path, "%s");
    if (pathParameters == null || countMatches != pathParameters.size()) {
      throw new IllegalArgumentException("Wrong number of path parameters");
    }
    return path.formatted(pathParameters.toArray(new Object[0]));
  }

  private String addPathParameter(String path, String pathParameter) {
    if (StringUtils.isBlank(pathParameter)) {
      return path;
    }
    StringBuilder pathBuilder = new StringBuilder(path);
    // get rid of leading slash
    String adjustedPathParameter = pathParameter;
    if (pathParameter.startsWith(SLASH)) {
      if (pathParameter.length() == 1) {
        adjustedPathParameter = "";
      } else {
        adjustedPathParameter = pathParameter.substring(1);
      }
    }
    if (!path.endsWith(SLASH)) {
      pathBuilder.append(SLASH);
    }
    pathBuilder.append(adjustedPathParameter);
    return pathBuilder.toString();
  }

  private <R> ServiceResponse<R> execute(
      HttpUriRequest request, Class<R> responseType, ResponseChecker responseChecker)
      throws RetryException, NoRetryException {
    ServiceResponse<R> serviceResponse = new ServiceResponse<>();
    try {
      HttpDestination destination =
          serviceEndpoint.getDestination() != null
              ? serviceEndpoint.getDestination()
              : ServiceDestinations.getDestination(serviceEndpoint.getDestinationName());
      var client = HttpClientFactory.getHttpClient(destination);
      logger.debug(
          "Service {} is called with url {} and path {}",
          serviceEndpoint.getDestinationName(),
          destination.getUri(),
          request.getURI());
      try (CloseableHttpResponse response = (CloseableHttpResponse) client.execute(request)) {
        int responseCode = response.getStatusLine().getStatusCode();
        serviceResponse.setHttpStatusCode(responseCode);
        serviceResponse.setHeaders(response.getAllHeaders());
        String returnedPayloadAsString = "";
        if (response.getEntity() != null && responseType != Void.class) {
          returnedPayloadAsString = EntityUtils.toString(response.getEntity());
          serviceResponse.setPayload(
              Optional.ofNullable(asResultType(returnedPayloadAsString, responseType)));
        }
        if (response.containsHeader(HttpHeaders.ETAG)) {
          serviceResponse.setETag(Optional.of(response.getLastHeader(HttpHeaders.ETAG).getValue()));
        }
        ResponseChecker.CheckResult checkResult = responseChecker.checkCode(responseCode);
        Optional<Exception> exception = checkResult.getException();
        if (checkResult.isTemporaryProblem()) {
          logger.debug(
              "A temporary problem was detected. Service call returned with code {}", responseCode);
          if (StringUtils.isNotEmpty(returnedPayloadAsString)) {
            logger.debug(
                "Service {} was called with url {} and path {} and returned payload {}",
                serviceEndpoint.getDestinationName(),
                destination.getUri(),
                request.getURI(),
                returnedPayloadAsString);
          }
          var requestedWaitTime = getRequestedWaitTime(responseCode, response);
          if (exception.isPresent()) {
            throw new RetryException(exception.get(), serviceResponse, requestedWaitTime);
          }
          // request a retry but the caller doesn't demand an exception
          throw new RetryException(serviceResponse, requestedWaitTime);
        }
        // an exception without retry
        if (exception.isPresent()) {
          logger.error(
              "A non temporary problem was detected. Service call returned with code {}",
              responseCode);
          if (StringUtils.isNotEmpty(returnedPayloadAsString)) {
            logger.debug(
                "Service {} was called with url {} and path {} and returned payload {}",
                serviceEndpoint.getDestinationName(),
                destination.getUri(),
                request.getURI(),
                returnedPayloadAsString);
          }
          throw new NoRetryException(exception.get(), serviceResponse);
        }
      }
    } catch (RetryException | NoRetryException exception) {
      throw exception;
    } catch (IOException ioException) {
      throw new RetryException(ioException, serviceResponse, null);
    } catch (Exception e) {
      throw new NoRetryException(e, serviceResponse);
    }
    return serviceResponse;
  }

  private Duration getRequestedWaitTime(int responseCode, CloseableHttpResponse response) {
    Duration requestedWaitTime = null;
    if (responseCode == HttpStatus.SC_TOO_MANY_REQUESTS) {
      var retryAfterHeader =
          response.containsHeader(RETRY_AFTER)
              ? response.getFirstHeader(RETRY_AFTER)
              : response.getFirstHeader(X_RATE_LIMIT_RETRY_AFTER);
      if (retryAfterHeader != null) {
        var retryAfterStr = retryAfterHeader.getValue();
        requestedWaitTime = getWaitTimeFromDateTime(retryAfterStr);
        if (requestedWaitTime == null) {
          requestedWaitTime = getRequestedWaitTimeFromInt(retryAfterStr);
        }
      }
    }
    return requestedWaitTime;
  }

  private static Duration getRequestedWaitTimeFromInt(String retryAfterStr) {
    try {
      return Duration.ofSeconds(Integer.parseInt(retryAfterStr));
    } catch (NumberFormatException e) {
      logger.debug("Wait time isn't specified as seconds", e);
      return null;
    }
  }

  private Duration getWaitTimeFromDateTime(String dateTime) {
    try {
      var formatter = DateTimeFormatter.RFC_1123_DATE_TIME.withLocale(Locale.US);
      var retryAfterDateTime = ZonedDateTime.parse(dateTime, formatter);
      ZonedDateTime now = ZonedDateTime.now(retryAfterDateTime.getZone());
      if (retryAfterDateTime.isBefore(now)) {
        return Duration.ofMillis(1);
      }
      return Duration.between(now, retryAfterDateTime);
    } catch (DateTimeParseException e) {
      logger.debug("Wait time isn't specified as date-time ", e);
      return null;
    }
  }

  @SuppressWarnings("unchecked")
  private <R> R asResultType(String payload, Class<R> responseType) {
    if (StringUtils.isEmpty(payload)) {
      payload = "{}";
    }
    if (responseType == String.class) {
      return (R) payload;
    }
    try {
      return objectMapper.readValue(payload, responseType);
    } catch (JsonProcessingException e) {
      return null;
    }
  }

  private HttpEntity asEntity(Object payload) throws ServiceException {

    try {
      if (payload instanceof String) {
        return new StringEntity((String) payload);
      } else {
        return new StringEntity(
            new ObjectMapper().writerWithDefaultPrettyPrinter().writeValueAsString(payload));
      }
    } catch (JsonProcessingException | UnsupportedEncodingException e) {
      throw new ServiceException(e, null);
    }
  }

  @SuppressWarnings("unchecked")
  <T extends Exception> T determineException(Exception exception, Class<T> exceptionClass) {
    T resultException = null;
    if (exceptionClass.isInstance(exception)) {
      resultException = (T) exception;
    } else if (exceptionClass.isInstance(exception.getCause())) {
      resultException = (T) exception.getCause();
    }
    return resultException;
  }
}
