/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.model;

import static com.sap.cds.impl.parser.ExpressionParser.parsePredicate;

import com.google.common.base.Strings;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExpression;
import com.sap.cds.ql.cqn.CqnInSubquery;
import com.sap.cds.ql.cqn.CqnMatchPredicate;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.util.ConstantLiteralSealingModifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * A {@code Privilege} specifies access rules for operations. A Where clause can be used to define
 * instance-based access.
 *
 * <p>grant: one or more operations (as a string or an array of strings) <br>
 * to: (optional) one or more user roles the privilege is granted to <br>
 * where: (optional) a condition that further restricts access
 *
 * <p>example: <code>
 *  { grant: ['READ','WRITE'], to: 'admin' }
 *  { grant: 'READ', where: 'buyer = $user' }
 * </code>
 */
public class Privilege {

  private static final CqnElementRef $USER_REF = CQL.get(CdsModelUtils.$USER);
  private static final Modifier LITERAL_SEALING_MODIFIER = new ConstantLiteralSealingModifier();
  private static final Modifier WHERE_MODIFIER = new UserIdNormalizationModifier();

  public enum PredefinedGrant {
    /** grant: '*' means all events */
    ALL("*"),

    /** grant. 'WRITE' means all standard events that have write semantic */
    WRITE("WRITE");

    private final String grant;

    PredefinedGrant(String grant) {
      this.grant = grant;
    }

    public boolean is(String grant) {
      return Privilege.is(this.grant, grant);
    }

    @Override
    public String toString() {
      return grant;
    }
  }

  /** Predefined roles that are handled specifically */
  public enum PredefinedRole {
    /** to: 'any' means all roles are accepted */
    ANY_USER("any"),

    /** to: 'authenticated-user' means all authenticated users are accepted */
    AUTHENTICATED_USER("authenticated-user"),

    /** to: 'system-user' is for technical users */
    SYSTEM_USER("system-user"),

    /** to: 'internal-user' is for internal technical users */
    INTERNAL_USER("internal-user");

    private final String role;

    PredefinedRole(String role) {
      this.role = role;
    }

    public boolean is(String role) {
      // TODO: are roles case insensitive ?
      return Privilege.is(this.role, role);
    }

    @Override
    public String toString() {
      return role;
    }
  }

  /** field 'grant:' */
  private List<String> grants = new ArrayList<>();

  /** field 'to:' */
  private List<String> roles = new ArrayList<>();

  // the predicate CQL string (only available in string expressions)
  private String whereCQL;

  // the parsed where predicate
  private CqnPredicate whereXpr;

  // the json representation of whereXpr
  private String cxnWhereCondition;

  public Privilege() {
    // use defaults
  }

  @SuppressWarnings("unchecked")
  public Privilege(Map<String, Object> privilege, String entityName) {
    Object grant = privilege.get("grant");
    if (grant instanceof String g) {
      addGrant(g);
    } else if (grant instanceof List g) {
      ((List<String>) g).forEach(this::addGrant);
    }

    Object to = privilege.get("to");
    if (to instanceof String t) {
      addRole(t);
    } else if (to instanceof List t) {
      ((List<String>) t).forEach(this::addRole);
    }

    readWhere(privilege, entityName);
  }

  public Privilege copy() {
    Privilege privilege = new Privilege();
    privilege.grants.addAll(this.getGrants());
    privilege.roles.addAll(this.getRoles());
    privilege.whereCQL = this.getWhereCQL();
    privilege.whereXpr = CQL.copy(this.whereXpr);
    privilege.cxnWhereCondition = this.cxnWhereCondition;
    return privilege;
  }

  public List<String> getGrants() {
    return Collections.unmodifiableList(grants);
  }

  public Privilege addGrant(PredefinedGrant grant) {
    return addGrant(grant.toString());
  }

  public Privilege addGrant(String grant) {
    Objects.requireNonNull(StringUtils.notEmpty(grant));

    grants.add(grant.trim());
    return this;
  }

  public List<String> getRoles() {
    return Collections.unmodifiableList(roles);
  }

  public Privilege addRole(PredefinedRole role) {
    return addRole(role.toString());
  }

  public Privilege addRole(String role) {
    Objects.requireNonNull(StringUtils.notEmpty(role));

    roles.add(role.trim());
    return this;
  }

  public String getWhereCQL() {
    return whereCQL;
  }

  public void setWhereCQL(String whereCQL) {
    this.whereCQL = whereCQL;
  }

  public CqnPredicate getWhereXpr() {
    return whereXpr;
  }

  // where.xpr property (where as expression)
  // compiler replaces $user with $user.id
  public void setWhereXpr(CqnPredicate where) {
    this.whereXpr = CQL.copy(where, WHERE_MODIFIER);
    this.cxnWhereCondition = whereXpr.toJson();
  }

  public boolean hasWhere() {
    return whereXpr != null;
  }

  public String getCxnWhereCondition() {
    return cxnWhereCondition;
  }

  // _where property (where as string)
  public void setCxnWhereCondition(String cxnWhereCondition) {
    CqnPredicate where = parsePredicate(cxnWhereCondition);
    this.whereXpr = CQL.copy(where, LITERAL_SEALING_MODIFIER);
    this.cxnWhereCondition = whereXpr.toJson();
  }

  public boolean hasWhereUsing(String xpr) {
    return Strings.nullToEmpty(cxnWhereCondition).contains(xpr);
  }

  public static boolean is(String a, String b) {
    return a == b || (a != null && b != null && a.equals(b));
  }

  private void readWhere(Map<String, Object> priv, String entityName) {
    Object where = priv.get("where");
    if (where != null) {
      if (where instanceof String w) {
        // where as string
        this.setWhereCQL(w);
        Object whereCxn = priv.get("_where");
        if (whereCxn != null) {
          this.setCxnWhereCondition(whereCxn.toString());
          return;
        }
      } else if (where instanceof CqnExpression xpr) {
        // where as expression
        this.setWhereXpr(xpr.asPredicate());
        return;
      }
      throw new ErrorStatusException(
          CdsErrorStatuses.INVALID_WHERE_CONDITION, where, entityName, "", "");
    }
  }

  static class UserIdNormalizationModifier extends ConstantLiteralSealingModifier {
    @Override
    public CqnValue ref(CqnElementRef ref) {
      if (ref.size() == 2 && ref.path().equals("$user.id")) {
        return $USER_REF;
      }
      return super.ref(ref);
    }

    @Override
    public CqnPredicate exists(Select<?> subQuery) {
      return CQL.exists(CQL.copy(subQuery, this));
    }

    @Override
    public CqnPredicate in(CqnInSubquery inSubquery) {
      throw new UnsupportedOperationException("Unsupported predicate: " + inSubquery);
    }

    @Override
    public CqnPredicate match(CqnMatchPredicate match) {
      Predicate filter = match.predicate().map(p -> CQL.copy(p, this)).orElse(null);

      return CQL.match(match.ref(), filter, match.quantifier());
    }
  }
}
