/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.utils.path;

import com.sap.cds.reflect.CdsDefinition;
import java.util.stream.Stream;

/**
 * A {@link CdsResourcePath} describes the hierarchical structure of a resource along with
 * additional meta information.
 */
public interface CdsResourcePath extends Comparable<CdsResourcePath> {

  /**
   * The path including the resource's base path.
   *
   * @return The path of the resource endpoint
   */
  String getPath();

  /**
   * If {@code true}, the endpoint should be exposed to public, i.e. without authentication
   *
   * @return {@code true} if public
   */
  boolean isPublic();

  /**
   * A {@link Stream} of events that are public even if {@code isPublic()==false}.
   *
   * @return The stream of public events
   */
  Stream<String> publicEvents();

  /**
   * A {@link Stream} of child paths.
   *
   * @return The child ResourcePaths
   */
  Stream<CdsResourcePath> subPaths();

  /**
   * Returns the CDS model element attached to the resource
   *
   * @return The CDS model element
   */
  CdsDefinition getCdsDefinition();

  /**
   * Checks if the referred {@link CdsDefinition} given with {@link
   * CdsResourcePath#getCdsDefinition()} is of given type.
   *
   * @param clazz The type to be checked
   * @param <T> The type parameter
   * @return {@code true} if the type matches the referred object
   */
  default <T extends CdsDefinition> boolean hasType(Class<T> clazz) {
    return clazz.isInstance(getCdsDefinition());
  }

  @Override
  default int compareTo(CdsResourcePath o) {
    return getPath().compareTo(o.getPath());
  }
}
