/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds;

import java.util.Map;

import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.Upsert;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnUpsert;

/**
 * The CdsDataStore is used to interact with a data source. It allows to execute
 * CDS QL {@link Select queries}, as well as {@link Insert insert},
 * {@link Upsert upsert}, {@link Update update} and {@link Delete delete}
 * statements.
 */
public interface CdsDataStore {

	/**
	 * Executes a {@link CqnSelect} statement with optional positional parameter
	 * values.
	 * 
	 * @param select      the CQN select statement to be executed
	 * @param paramValues the positional parameter values
	 * @return the {@link Result} of the query
	 */
	Result execute(CqnSelect select, Object... paramValues);

	/**
	 * Executes a {@link CqnSelect} statement with named parameter values.
	 * 
	 * @param select      the CQN select statement to be executed
	 * @param namedValues the named parameter values
	 * @return the {@link Result} of the query
	 */
	Result execute(CqnSelect select, Map<String, Object> namedValues);

	/**
	 * Executes a {@link CqnInsert} statement.
	 * 
	 * @param insert the CQN insert statement to be executed
	 * @return the {@link Result} of the insert
	 */
	Result execute(CqnInsert insert);

	/**
	 * Executes a {@link CqnUpsert} statement.
	 * 
	 * @param upsert the CQN upsert statement to be executed
	 * @return the {@link Result} of the upsert
	 */
	Result execute(CqnUpsert upsert);

	/**
	 * Executes a {@link CqnUpdate} statement with optional positional parameter
	 * values.
	 * 
	 * @param update      the CQN update statement to be executed
	 * @param paramValues the positional parameter values
	 * @return the {@link Result} of the update
	 */
	Result execute(CqnUpdate update, Object... paramValues);

	/**
	 * Executes a {@link CqnUpdate} statement with named parameter values.
	 * 
	 * @param update      the CQN update statement to be executed
	 * @param namedValues the named parameter values
	 * @return the {@link Result} of the update
	 */
	Result execute(CqnUpdate update, Map<String, Object> namedValues);

	/**
	 * Executes a {@link CqnUpdate} statement as batch with the given named
	 * parameter values.
	 * 
	 * @param update    the CQN update statement to be executed
	 * @param valueSets the named parameter value sets
	 * @return the {@link Result} of the batch update
	 */
	Result execute(CqnUpdate update, Iterable<Map<String, Object>> valueSets);

	/**
	 * Executes a {@link CqnDelete} statement with optional positional parameter
	 * values.
	 * 
	 * @param delete      the CQN delete statement to be executed
	 * @param paramValues the positional parameter values
	 * @return the {@link Result} of the delete
	 */
	Result execute(CqnDelete delete, Object... paramValues);

	/**
	 * Executes a {@link CqnDelete} statement with named parameter values.
	 * 
	 * @param delete      the CQN delete statement to be executed
	 * @param namedValues the named parameter values
	 * @return the {@link Result} of the delete
	 */
	Result execute(CqnDelete delete, Map<String, Object> namedValues);

	/**
	 * Executes a {@link CqnDelete} statement as batch with the given named
	 * parameter values.
	 * 
	 * @param delete    the CQN delete statement to be executed
	 * @param valueSets the named parameter value sets
	 * @return the {@link Result} of the batch delete
	 */
	Result execute(CqnDelete delete, Iterable<Map<String, Object>> valueSets);

	SessionContext getSessionContext();

	void setSessionContext(SessionContext session);
}
