/*******************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds;

import static com.sap.cds.Cds4jServiceLoader.load;
import static java.util.Arrays.asList;
import static java.util.Arrays.stream;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.google.common.annotations.Beta;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.reflect.CdsStructuredType;

public abstract class ResultBuilder {

	private static ResultBuilder builder() {
		return load(ResultBuilder.class);
	}

	private static ResultBuilder builder(List<? extends Map<String, ?>> rows) {
		return builder().rows(rows);
	}

	public static ResultBuilder selectedRows(List<? extends Map<String, ?>> rows) {
		return builder(rows).rowCount(new int[] { rows.size() });
	}

	public static ResultBuilder insertedRows(List<? extends Map<String, ?>> rows) {
		int[] rowCount = new int[rows.size()];
		Arrays.fill(rowCount, 1);
		return builder(rows).rowCount(rowCount);
	}

	public static ResultBuilder updatedRows(int updateCount, Map<String, ?> updateData) {
		return updatedRows((long) updateCount, updateData);
	}

	public static ResultBuilder updatedRows(long updateCount, Map<String, ?> updateData) {
		if (updateCount == 0) {
			return builder();
		}
		return builder(asList(updateData)).rowCount(new long[] { updateCount });
	}

	public static ResultBuilder updatedRows(int[] updateCount, Map<String, ?> updateData) {
		return updatedRows(longArray(updateCount), updateData);
	}

	public static ResultBuilder updatedRows(long[] updateCount, Map<String, ?> updateData) {
		if (stream(updateCount).anyMatch(c -> c > 0)) {
			int length = updateCount.length;
			List<Map<String, ?>> rows = new ArrayList<>(length);
			for (int i = 0; i < length; i++) {
				rows.add(updateCount[i] > 0 ? updateData : new HashMap<>());
			}
			return builder(rows).rowCount(updateCount);
		}
		return builder();
	}

	public static ResultBuilder updatedRows(int[] updateCount, List<? extends Map<String, ?>> updateData) {
		return updatedRows(longArray(updateCount), updateData);
	}

	private static long[] longArray(int[] updateCount) {
		return stream(updateCount).asLongStream().toArray();
	}

	public static ResultBuilder updatedRows(long[] updateCount, List<? extends Map<String, ?>> updateData) {
		int size = updateData.size();
		if (size == 1) {
			return updatedRows(updateCount, updateData.get(0));
		}
		if (size != updateCount.length) {
			throw new CdsDataException("Update count does not match batch size");
		}
		List<Map<String, ?>> rows = new ArrayList<>(size);
		for (int i = 0; i < size; i++) {
			rows.add(updateCount[i] > 0 ? updateData.get(i) : new HashMap<>());
		}

		return builder(rows).rowCount(updateCount);
	}

	public static ResultBuilder deletedRows(int deleteCount) {
		return deletedRows((long) deleteCount);
	}

	public static ResultBuilder deletedRows(long deleteCount) {
		return deletedRows(new long[] { deleteCount });
	}

	public static ResultBuilder deletedRows(int[] deleteCount) {
		return builder().rowCount(deleteCount);
	}

	public static ResultBuilder deletedRows(long[] deleteCount) {
		return builder().rowCount(deleteCount);
	}

	protected ResultBuilder rows(List<? extends Map<String, ?>> rows) {
		return rows(rows.stream());
	}

	protected abstract ResultBuilder rows(Stream<? extends Map<String, ?>> rows);

	protected abstract ResultBuilder rowCount(int[] rowCount);

	protected abstract ResultBuilder rowCount(long[] rowCount);

	@Beta
	public abstract ResultBuilder rowType(CdsStructuredType rowType);

	public abstract ResultBuilder inlineCount(long inlineCount);

	public abstract Result result();

	public abstract long rowCount();

	public abstract long inlineCount();

}
