/*******************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql;

import static java.util.Arrays.asList;

import java.time.Instant;
import java.util.Collection;
import java.util.List;

import com.sap.cds.ql.cqn.CqnComparisonPredicate;
import com.sap.cds.ql.cqn.CqnConnectivePredicate;
import com.sap.cds.ql.cqn.CqnConnectivePredicate.Operator;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnModifier;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStar;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnValue;

public interface CQL {

	/**
	 * Creates a copy of the given predicate.
	 * 
	 * @param pred the predicate to be copied
	 * @return the copy
	 */
	static Predicate copy(CqnPredicate pred) {
		return CDS.QL.builder.copy(pred);
	}

	/**
	 * Creates a modified copy of the given predicate.
	 * 
	 * @param pred     the predicate to be copied
	 * @param modifier the modifier for altering the copy
	 * @return the modified copy
	 */
	static Predicate copy(CqnPredicate pred, CqnModifier modifier) {
		return CDS.QL.builder.copy(pred, modifier);
	}

	/**
	 * Creates a copy of the given CQN statement.
	 * 
	 * @param <S>       the type of CqnStatement
	 * @param statement the CQN statement to be copied
	 * @return the copy
	 */
	static <S extends CqnStatement> S copy(S statement) {
		return CDS.QL.builder.copy(statement);
	}

	/**
	 * Creates a modified copy of the given CQN statement.
	 * 
	 * @param <S>       the type of CqnStatement
	 * @param statement the CQN statement to be copied
	 * @param modifier  the modifier for altering the copy
	 * @return the modified copy
	 */
	static <S extends CqnStatement> S copy(S statement, CqnModifier modifier) {
		return CDS.QL.builder.copy(statement, modifier);
	}

	/**
	 * Creates a CDS QL function call that is send to the underlying data store.
	 * 
	 * @param <T>  the return type of the function call
	 * @param name the name of the function to be called
	 * @param args the arguments of the function
	 * @return the CDS QL function call
	 */
	static <T> FunctionCall<T> func(String name, CqnValue... args) {
		return func(name, asList(args));
	}

	/**
	 * Creates a CDS QL function call that is send to the underlying data store.
	 * 
	 * @param <T>  the return type of the function call
	 * @param name the name of the function to be called
	 * @param args the arguments of the function
	 * @return the CDS QL function call
	 */
	static <T> FunctionCall<T> func(String name, List<? extends CqnValue> args) {
		return CDS.QL.create.func(name, args);
	}

	/**
	 * Creates a CDS QL function call that is send to the underlying data store.
	 * 
	 * @param name the name of the function to be called
	 * @param args the arguments of the function
	 * @return the CDS QL function call
	 */
	static BooleanFunction booleanFunc(String name, List<? extends CqnValue> args) {
		return CDS.QL.create.booleanFunc(name, args);
	}

	/**
	 * Creates a CDS QL {@link Literal} to be used in queries.
	 * 
	 * @param <T>   the type of the literal
	 * @param value the value of the literal
	 * @return the literal
	 */
	static <T> Literal<T> literal(T value) {
		return CDS.QL.create.literal(value);
	}

	/**
	 * Negates a given {@link CqnPredicate}.
	 * 
	 * @param predicate the {@code CqnPredicate} to be negated
	 * @return the negated predicate
	 */
	static Predicate not(CqnPredicate predicate) {
		return CDS.QL.create.not(predicate);
	}

	/**
	 * Creates a conjunction of two given predicates
	 * 
	 * @param p1 1st operand of the conjunction
	 * @param p2 2nd operand of the conjunction
	 * @return the conjunction
	 */
	static Predicate and(CqnPredicate p1, CqnPredicate p2) {
		return and(asList(p1, p2));
	}

	/**
	 * Creates a disjunction of two given predicates
	 * 
	 * @param p1 1st operand of the disjunction
	 * @param p2 2nd operand of the disjunction
	 * @return the disjunction
	 */
	static Predicate or(CqnPredicate p1, CqnPredicate p2) {
		return or(asList(p1, p2));
	}

	/**
	 * Creates a conjunction of given predicates
	 * 
	 * @param predicates the predicates to connect
	 * @return the conjunction
	 */
	static Predicate and(Iterable<? extends CqnPredicate> predicates) {
		return connect(Operator.AND, predicates);
	}

	/**
	 * Creates a disjunction of given predicates
	 * 
	 * @param predicates the predicates to connect
	 * @return the disjunction
	 */
	static Predicate or(Iterable<? extends CqnPredicate> predicates) {
		return connect(Operator.OR, predicates);
	}

	/**
	 * Creates a logical connection of given predicates {@link CqnPredicate}.
	 * 
	 * @param operator   operator to connect the predicates with
	 * @param predicates the predicates to connect
	 * @return the logical connection
	 */
	static Predicate connect(CqnConnectivePredicate.Operator operator, Iterable<? extends CqnPredicate> predicates) {
		return CDS.QL.create.connect(operator, predicates);
	}

	/**
	 * Creates a CDS QL positional {@link Parameter} to be used in queries.
	 * 
	 * @param <T> the type of the parameter
	 * @return the parameter
	 */
	static <T> Parameter<T> param() {
		return CDS.QL.create.param();
	}

	/**
	 * Creates a CDS QL named {@link Parameter} to be used in queries.
	 * 
	 * @param <T>  the type of the parameter
	 * @param name the parameter name
	 * @return the parameter
	 */
	static <T> Parameter<T> param(String name) {
		return CDS.QL.create.param(name);
	}

	/**
	 * Creates a CQN plain value, which is directly send to the underlying data
	 * store.
	 * 
	 * @param value the value content
	 * @return the plain value
	 */
	static Value<?> plain(String value) {
		return CDS.QL.create.plain(value);
	}

	/**
	 * Creates a count all function call.
	 * 
	 * @return the count value
	 */
	static Value<Long> count() {
		return func("COUNT").type(Long.class);
	}

	/**
	 * Creates a count function call on the given element.
	 * 
	 * @param element the ref to the element to be counted
	 * @return the count value
	 */
	static Value<Long> count(CqnElementRef element) {
		return func("COUNT", element).type(Long.class);
	}

	/**
	 * Creates a CDS QL expression for converting the given string value to lower
	 * case using the rules of the underlying data store.
	 * 
	 * @param val the String value to be converted
	 * @return the CDS QL expression for lower case conversion
	 */
	static Value<String> toLower(Value<String> val) {
		return CDS.QL.create.toLower(val);
	}

	/**
	 * Creates a CDS QL expression for converting the given string value to lower
	 * case using the rules of the underlying data store.
	 * 
	 * @param val the String value to be converted
	 * @return the CDS QL expression for lower case conversion
	 */
	static Value<String> toLower(String val) {
		return CDS.QL.create.toLower(literal(val));
	}

	/**
	 * Creates a CDS QL expression for converting the given string value to upper
	 * case using the rules of the underlying data store.
	 * 
	 * @param val the String value to be converted
	 * @return the CDS QL expression for upper case conversion
	 */
	static Value<String> toUpper(Value<String> val) {
		return CDS.QL.create.toUpper(val);
	}

	/**
	 * Creates a CDS QL expression for converting the given string value to upper
	 * case using the rules of the underlying data store.
	 * 
	 * @param val the String value to be converted
	 * @return the CDS QL expression for upper case conversion
	 */
	static Value<String> toUpper(String val) {
		return CDS.QL.create.toUpper(literal(val));
	}

	/**
	 * Creates a {@link StructuredType} representing a CDS entity.
	 * 
	 * @param qualifiedName the qualified name of the CDS entity
	 * @return the {@code StructuredType} representing the CDS entity
	 */
	static StructuredType<?> entity(String qualifiedName) {
		return CDS.QL.create.entity(qualifiedName);
	}

	/**
	 * Creates a {@link StructuredType} representing a path from a CDS entity to
	 * another entity.
	 * 
	 * @param path a String of association element names separated by '.'
	 * @return the {@code StructuredType} representing the path to a CDS entity
	 */
	static StructuredType<?> to(String path) {
		return CDS.QL.create.to(path);
	}

	/**
	 * Creates a {@link StructuredType} representing a path from a CDS entity to
	 * another entity.
	 * 
	 * @param segments the segments of the path
	 * @return the {@code StructuredType} representing the path to a CDS entity
	 */
	static StructuredType<?> to(List<? extends Segment> segments) {
		return CDS.QL.create.to(segments);
	}

	/**
	 * Creates an {@link ElementRef} representing a path from a CDS entity to an
	 * element of this or another entity.
	 * 
	 * @param <T>  the type of the element
	 * @param path a String of element names separated by '.', where all except the
	 *             last segment represent association elements.
	 * @return the {@code ElementRef} representing the path to the CDS element
	 */
	static <T> ElementRef<T> get(String path) {
		return CDS.QL.create.get(path);
	}

	/**
	 * Creates an {@link ElementRef} representing a path from a CDS entity to an
	 * element of this or another entity.
	 * 
	 * @param <T>      the type of the element
	 * @param segments the segments of the path
	 * @return the {@code ElementRef} representing the path to the CDS element
	 */
	static <T> ElementRef<T> get(List<? extends Segment> segments) {
		return CDS.QL.create.get(segments);
	}

	/**
	 * Creates a segment of a structured type or element reference.
	 * 
	 * @param id the id of the segment
	 * @return the segment
	 */
	static RefSegment refSegment(String id) {
		return CDS.QL.create.refSegment(id);
	}

	/**
	 * Creates a list of reference segments.
	 * 
	 * @param segmentIds the ids of the segments
	 * @return the segments
	 */
	static List<RefSegment> refSegments(List<String> segmentIds) {
		return CDS.QL.create.refSegments(segmentIds);
	}

	/**
	 * Creates a comparison predicate.
	 * 
	 * @param lhs the left-hand side of the comparison
	 * @param op  the comparison operator
	 * @param rhs the right-hand side of the comparison
	 * @return the comparison predicate
	 */
	static Predicate comparison(CqnValue lhs, CqnComparisonPredicate.Operator op, CqnValue rhs) {
		return CDS.QL.create.comparison(lhs, op, rhs);
	}

	/**
	 * Creates an in predicate.
	 * 
	 * @param lhs    the value to be checked if it's contained in the collection
	 * @param values the collection of values
	 * @return the in predicate
	 */
	static Predicate in(CqnValue lhs, Collection<? extends CqnValue> values) {
		return CDS.QL.create.in(lhs, values);
	}

	/**
	 * Creates a search predicate.
	 * 
	 * @param term the search term
	 * @return the search predicate
	 */
	static Predicate search(String term) {
		return CDS.QL.create.search(term);
	}

	/**
	 * Creates an exists predicate that tests if a given subquery returns any row
	 * 
	 * @param subQuery the subquery that performs the existence test
	 * @return the exists predicate
	 */
	static Predicate exists(CqnSelect subQuery) {
		return CDS.QL.create.exists(subQuery);
	}

	/**
	 * Creates an {@link ElementRef} to be used in queries to represent $now.
	 * 
	 * @return the now value
	 */
	static Value<Instant> now() {
		return CDS.QL.create.now().type(Instant.class);
	}

	/**
	 * Creates an {@link ElementRef} on the available $validFrom value.
	 * 
	 * @return the validFrom value
	 */
	static Value<Instant> validFrom() {
		return CDS.QL.create.validFrom().type(Instant.class);
	}

	/**
	 * Creates an {@link ElementRef} on the available $validTo value.
	 * 
	 * @return the validTo value
	 */
	static Value<Instant> validTo() {
		return CDS.QL.create.validTo().type(Instant.class);
	}

	/**
	 * Creates an {@link ElementRef} on the available $user.locale value.
	 * 
	 * @return the locale string value
	 */
	static Value<String> userLocale() {
		return CDS.QL.create.userLocale().type(String.class);
	}

	/**
	 * Creates an {@link ElementRef} on the available $user.id value.
	 * 
	 * @return the user.id value
	 */
	static Value<String> userId() {
		return CDS.QL.create.userId().type(String.class);
	}

	static CqnStar star() {
		return CqnStar.star();
	}

}
