/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql;

import java.util.Map;
import java.util.function.Function;
import java.util.function.UnaryOperator;

import com.sap.cds.CdsDataStore;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.reflect.CdsEntity;

/**
 * The Insert builder allows to construct CDS QL insert statements, which can be
 * executed via the {@link CdsDataStore}.
 */
public interface Insert extends CqnInsert {

	/**
	 * Creates an insert statement to insert entries into a specified entity set.
	 *
	 * @param qualifiedName the fully qualified name of the CDS entity set
	 * @return the insert statement
	 */
	static Insert into(String qualifiedName) {
		return CDS.QL.builder.insert(qualifiedName, e -> e);
	}

	/**
	 * Creates an insert statement to insert entries into a specified entity set.
	 *
	 * @param ref the ref to the entity
	 * @return the insert statement
	 */
	public static Insert into(CqnStructuredTypeRef ref) {
		return CDS.QL.builder.insert(ref);
	}

	/**
	 * Creates an insert statement to insert entries into a specified entity set.
	 *
	 * @param        <E> the type of the entity set
	 * @param entity the structured type representing the entity set
	 * @return the insert statement
	 */
	public static <E extends StructuredType<?>> Insert into(E entity) {
		return CDS.QL.builder.insert(entity);
	}

	/**
	 * Creates an insert statement to insert entries into a specified entity set.
	 *
	 * @param entityName the root entity name of the path expression
	 * @param path       a path expression navigating from the root entity to the
	 *                   target entity of the insert statement
	 * @return the insert statement
	 */
	static Insert into(String entityName, UnaryOperator<StructuredType<?>> path) {
		return CDS.QL.builder.insert(entityName, path);
	}

	/**
	 * Creates an insert statement to insert entries into a specified entity set.
	 *
	 * @param        <E> the type of the entity set
	 * @param entity the static model representation of the entity set
	 * @return the insert statement
	 */
	static <E extends StructuredType<E>> Insert into(Class<E> entity) {
		return CDS.QL.builder.insert(entity, e -> e);
	}

	/**
	 * Creates an insert statement to insert entries into a specified entity set.
	 *
	 * @param        <R> the type of the root entity
	 * @param        <T> the type of the target entity
	 * @param entity the static model representation of the entity set
	 * @param path   a path expression navigating from the root entity to the target
	 *               entity of the insert statement
	 * @return the insert statement
	 */
	static <R extends StructuredType<R>, T extends StructuredType<T>> Insert into(Class<R> entity,
			Function<R, T> path) {
		return CDS.QL.builder.insert(entity, path);
	}

	/**
	 * Creates an insert statement to insert entries into a specified entity set.
	 *
	 * @param entity the model representation of the entity set obtained by
	 *               reflection
	 * @return the insert statement
	 * @see com.sap.cds.reflect.CdsModel#findEntity(String)
	 * @see com.sap.cds.reflect.CdsModel#entities()
	 */
	static Insert into(CdsEntity entity) {
		return CDS.QL.builder.insert(entity, e -> e);
	}

	/**
	 * Copies the given {@link CqnInsert} into an {@link Insert} builder.
	 *
	 * @param insert the {@code CqnInsert} to be copied
	 * @return the modifiable insert statement copy
	 */
	static Insert copy(CqnInsert insert) {
		return CDS.QL.builder.copy(insert);
	}

	/**
	 * Creates an insert statement from a given CQN String.
	 *
	 * @param cqnInsert the CQN representation of the insert statement
	 * @return the insert statement
	 */
	static Insert cqn(String cqnInsert) {
		return CDS.QL.parser.insert(cqnInsert);
	}

	/**
	 * Sets the entries to be inserted into the entity set.
	 *
	 * @param entries a collection of data where every element is the canonical
	 *                representation
	 * @return the insert statement
	 */
	Insert entries(Iterable<? extends Map<String, ?>> entries);

	/**
	 * Add a single entry to be inserted into the entity set. The data is given as a
	 * {@literal Map<String, Object>} that maps element names of the target entity
	 * set to the values to be inserted.
	 * <p>
	 * The value can be deeply structured to represent a structured document:
	 * <ul>
	 * <li>for single-valued relationships the value of the element is of type
	 * {@literal Map<String, Object>}</li>
	 * <li>for collection-valued relationships the value of the element is of type
	 * {@literal List<Map<String, Object>>}</li>
	 * </ul>
	 *
	 * @param entry the data to be inserted as a map
	 *
	 * @return the insert statement
	 */
	Insert entry(Map<String, ?> entry);

	/**
	 * Add a single entry to be inserted into the entity set.
	 * <p>
	 * The value can be deeply structured to represent a structured document:
	 * <ul>
	 * <li>for single-valued relationships the value of the element is of type
	 * {@literal Map<String, Object>}</li>
	 * <li>for collection-valued relationships the value of the element is of type
	 * {@literal List<Map<String, Object>>}</li>
	 * </ul>
	 *
	 * @param elementName the element name of the target entity for the value to be
	 *                    inserted
	 * @param value       the data to be inserted for the element
	 *
	 * @return the insert statement
	 */
	Insert entry(String elementName, Object value);

}
