/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql;

import java.util.Map;
import java.util.function.Function;
import java.util.function.UnaryOperator;

import com.sap.cds.CdsDataStore;
import com.sap.cds.ql.cqn.CqnParameter;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.reflect.CdsEntity;

/**
 * The Update Builder allows to construct CDS QL update statements, which can be
 * executed via the {@link CdsDataStore}.
 *
 * @param <T> the type of the entity set targeted by this update statement
 */
public interface Update<T extends StructuredType<?>> extends CqnUpdate {

	/**
	 * Creates an update statement to update entries of a specified entity set.
	 *
	 * @param qualifiedName the fully qualified name of the CDS entity set
	 * @return the update statement
	 */
	static Update<StructuredType<?>> entity(String qualifiedName) {
		return entity(qualifiedName, e -> e);
	}

	/**
	 * Creates an update statement to update entries of a specified entity set.
	 *
	 * @param ref the ref to the entity
	 * @return the update statement
	 */
	static Update<StructuredType<?>> entity(CqnStructuredTypeRef ref) {
		return CDS.QL.builder.update(ref);
	}

	/**
	 * Creates an update statement to update entries from a specified entity set.
	 *
	 * @param        <T> the type of the entity set
	 * @param entity the structured type representing the entity set
	 * @return the update statement
	 */
	public static <T extends StructuredType<?>> Update<T> entity(T entity) {
		return CDS.QL.builder.update(entity);
	}

	/**
	 * Creates an update statement to update entries of a specified entity set.
	 *
	 * @param qualifiedName the fully qualified name of the CDS entity set
	 * @param path          a path expression navigating from the root entity to the
	 *                      target entity of the update statement
	 * @return the update statement
	 */
	static Update<StructuredType<?>> entity(String qualifiedName, UnaryOperator<StructuredType<?>> path) {
		return CDS.QL.builder.update(qualifiedName, path);
	}

	/**
	 * Creates an update statement to update data for a specified entity set.
	 *
	 * @param        <T> the type of the entity set
	 * @param entity the static model representation of the entity set
	 * @return the update statement
	 */
	static <T extends StructuredType<T>> Update<T> entity(Class<T> entity) {
		return entity(entity, e -> e);
	}

	/**
	 * Creates an update statement to update data for a specified entity set.
	 *
	 * @param        <R> the type of the root entity
	 * @param        <T> the type of the target entity
	 * @param entity the static model representation of the entity set
	 * @param path   a path expression navigating from the root entity to the target
	 *               entity of the update statement
	 * @return the update statement
	 */
	static <R extends StructuredType<R>, T extends StructuredType<T>> Update<T> entity(Class<R> entity,
			Function<R, T> path) {
		return CDS.QL.builder.update(entity, path);
	}

	/**
	 * Creates an update statement to update entries of a specified entity set.
	 *
	 * @param entity the model representation of the entity set obtained by
	 *               reflection
	 * @param path   a path expression navigating from the root entity to the target
	 *               entity of the update statement
	 * @return the update statement
	 * @see com.sap.cds.reflect.CdsModel#findEntity(String)
	 * @see com.sap.cds.reflect.CdsModel#entities()
	 */
	static Update<StructuredType<?>> entity(CdsEntity entity, UnaryOperator<StructuredType<?>> path) {
		return CDS.QL.builder.update(entity, path);
	}

	/**
	 * Creates an update statement to update data for a specified entity set.
	 *
	 * @param entity the model representation of the entity set obtained by
	 *               reflection
	 * @return the update statement
	 * @see com.sap.cds.reflect.CdsModel#findEntity(String)
	 * @see com.sap.cds.reflect.CdsModel#entities()
	 */
	static Update<StructuredType<?>> entity(CdsEntity entity) {
		return entity(entity, e -> e);
	}

	/**
	 * Copies the given {@link CqnUpdate} into an {@link Update} builder.
	 *
	 * @param update the {@code CqnUpdate} to be copied
	 * @return the modifiable update statement copy
	 */
	static Update<StructuredType<?>> copy(CqnUpdate update) {
		return CDS.QL.builder.copy(update);
	}

	/**
	 * Creates an update statement to update data for a specified entity set.
	 *
	 * @param cqnUpdate the CQN representation of the update statement
	 * @return the update statement
	 */
	static Update<StructuredType<?>> cqn(String cqnUpdate) {
		return CDS.QL.parser.update(cqnUpdate);
	}

	/**
	 * Sets the batch data to be updated by the update statement. The data is given
	 * as a collection of maps from element names of the target entity set to their
	 * new values. The data must contain the entities' keys.
	 *
	 * @param entries a collection of data to be updated given as a map from element
	 *                name to new value
	 * @return the update statement
	 */
	Update<T> entries(Iterable<? extends Map<String, ?>> entries);

	/**
	 * Sets the values to be updated by the update statement. The data is given as a
	 * map of element names of the target entity set to their new values.
	 *
	 * @param data the data to be updated as a map from element name to new value
	 * @return the update statement
	 */
	Update<T> data(Map<String, ?> data);

	/**
	 * Adds an element to be updated to the update statement.
	 *
	 * @param elementName the element name of the target entity
	 * @param value       the new value of the element
	 * @return the update statement
	 */
	Update<T> data(String elementName, Object value);

	/**
	 * Sets the where condition of the update statement.
	 *
	 * @param predicate a {@link CqnPredicate}
	 * @return the update statement
	 */
	Update<T> where(CqnPredicate predicate);

	/**
	 * Sets the where condition of the update statement. The where condition is
	 * provided as a function that accepts a model object representing the entity
	 * targeted by the update and returning the where condition.
	 *
	 * @param predicate a {@link Function} providing the where condition
	 * @return the update statement
	 */
	Update<T> where(Function<T, CqnPredicate> predicate);

	/**
	 * Sets the where condition of the update statement. The where condition is
	 * computed from a map of element names of the target entity set to their
	 * values, or a {@link CqnParameter}. The map entries are transformed into
	 * comparison predicates and joined via <b>and</b>.
	 *
	 * @param values the element name to value map defining the where condition
	 * @return the update statement
	 */
	Update<T> matching(Map<String, ?> values);

	/**
	 * Sets the where condition of the update statement, requiring that the value of
	 * the single key element of the target entity set is equal to the given
	 * idValue.
	 *
	 * @param idValue the value of the target entity's key element
	 * @return the update statement
	 */
	Update<T> byId(Object idValue);

}
