/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.ql;

import java.util.List;

import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsBaseType;

public interface Value<T> extends CqnValue, Orderable {

	/**
	 * Creates an expression that adds a given summand to this numeric value
	 * 
	 * @param v the summand
	 * @return expression for the addition
	 */
	Value<T> plus(T v);

	/**
	 * Creates an expression that adds a given summand to this numeric value
	 * 
	 * @param v the summand
	 * @return expression for the addition
	 */
	Value<T> plus(Value<T> v);

	/**
	 * Creates an expression that subtracts a given subtrahend from this numeric
	 * value
	 * 
	 * @param v the subtrahend
	 * @return expression for the subtraction
	 */
	Value<T> minus(T v);

	/**
	 * Creates an expression that subtracts a given subtrahend from this numeric
	 * value
	 * 
	 * @param v the subtrahend
	 * @return expression for the subtraction
	 */
	Value<T> minus(Value<T> v);

	/**
	 * Creates an expression that multiplies this numeric value with a given factor
	 * 
	 * @param v the factor
	 * @return expression for the multiplication
	 */
	Value<T> times(T v);

	/**
	 * Creates an expression that multiplies this numeric value with a given factor
	 * 
	 * @param v the factor
	 * @return expression for the multiplication
	 */
	Value<T> times(Value<T> v);

	/**
	 * Creates an expression that divides this numeric value by a given divisor
	 * 
	 * @param v the divisor
	 * @return expression for the division
	 */
	Value<T> dividedBy(T v);

	/**
	 * Creates an expression that divides this numeric value by a given divisor
	 * 
	 * @param v the divisor
	 * @return expression for the division
	 */
	Value<T> dividedBy(Value<T> v);

	/**
	 * Creates an expression for substring extraction from this value. Extracts a
	 * substring of given length starting at a specified position. First position is
	 * 0.
	 * 
	 * @param start  the start position
	 * @param length the number of characters that you want to extract
	 * @return expression for substring extraction
	 */
	Value<String> substring(Value<Integer> start, Value<Integer> length);

	/**
	 * Creates an expression for substring extraction from this string value.
	 * Extracts a substring of given length starting at a specified position. First
	 * position is 0.
	 * 
	 * @param start  the start position
	 * @param length the number of characters that you want to extract
	 * @return expression for substring extraction
	 */
	Value<String> substring(int start, int length);

	/**
	 * Creates an expression for substring extraction from this string value.
	 * Extracts a substring starting at a specified position extending to the end of
	 * the string. First position is 0.
	 * 
	 * @param start the start position
	 * @return the expression for substring extraction
	 */
	Value<String> substring(Value<Integer> start);

	/**
	 * Creates an expression for substring extraction from this string value.
	 * Extracts a substring starting at a specified position extending to the end of
	 * the string. First position is 0.
	 * 
	 * @param start the start position
	 * @return the expression for substring extraction
	 */
	Value<String> substring(int start);

	/**
	 * Creates an expression for converting this string value to upper case using
	 * the rules of the underlying data store.
	 * 
	 * @return the String value in UpperCase
	 */
	Value<String> toUpper();

	/**
	 * Creates an expression for converting this string value to lower case using
	 * the rules of the underlying data store.
	 * 
	 * @return the String value in LowerCase
	 */
	Value<String> toLower();

	/**
	 * Creates a predicate that tests if this value is null
	 * 
	 * @return the predicate for the null test
	 */
	Predicate isNull();

	/**
	 * Creates a predicate that tests if this value is not null
	 * 
	 * @return the predicate for the not null test
	 */
	Predicate isNotNull();

	/**
	 * Creates a comparison predicate that tests if this value is equal to a given
	 * value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate eq(T v);

	/**
	 * Creates a comparison predicate that tests if this value is equal to a given
	 * value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate eq(Value<T> v);

	/**
	 * Creates a comparison predicate that tests if this value is not equal to a
	 * given value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate ne(T v);

	/**
	 * Creates a comparison predicate that tests if this value is not equal to a
	 * given value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate ne(Value<T> v);

	/**
	 * Creates a comparison predicate that tests if this value is greater than a
	 * given value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate gt(T v);

	/**
	 * Creates a comparison predicate that tests if this value is greater than a
	 * given value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate gt(Value<T> v);

	/**
	 * Creates a comparison predicate that tests if this value is greater than or
	 * equal to a given value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate ge(T v);

	/**
	 * Creates a comparison predicate that tests if this value is greater than or
	 * equal to a given value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate ge(Value<T> v);

	/**
	 * Creates a comparison predicate that tests if this value is less than a given
	 * value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate lt(T v);

	/**
	 * Creates a comparison predicate that tests if this value is less than a given
	 * value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate lt(Value<T> v);

	/**
	 * Creates a comparison predicate that tests if this value is less than or equal
	 * to a given value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate le(T v);

	/**
	 * Creates a comparison predicate that tests if this value is less than or equal
	 * to a given value
	 * 
	 * @param v the value to compare with
	 * @return the comparison predicate
	 */
	Predicate le(Value<T> v);

	/**
	 * Creates a between predicate that tests if this value is within a closed
	 * interval with given bounds
	 * 
	 * @param low  the lower bound
	 * @param high the upper bound
	 * @return the between predicate
	 */
	Predicate between(T low, T high);

	/**
	 * Creates a between predicate that tests if this value is within a closed
	 * interval with given bounds
	 * 
	 * @param low  the lower bound
	 * @param high the upper bound
	 * @return the between predicate
	 */
	Predicate between(Value<T> low, T high);

	/**
	 * Creates a between predicate that tests if this value is within a closed
	 * interval with given bounds
	 * 
	 * @param low  the lower bound
	 * @param high the upper bound
	 * @return the between predicate
	 */
	Predicate between(T low, Value<T> high);

	/**
	 * Creates a between predicate that tests if this value is within a closed
	 * interval with given bounds
	 * 
	 * @param low  the lower bound
	 * @param high the upper bound
	 * @return the between predicate
	 */
	Predicate between(Value<T> low, Value<T> high);

	/**
	 * Creates an IN predicate that tests if this value is equal to any value in a
	 * given list
	 * 
	 * @param vs the given values
	 * @return the IN predicate for the in list test
	 */
	Predicate in(@SuppressWarnings("unchecked") T... vs);

	/**
	 * Creates an IN predicate that tests if this value is equal to any value in a
	 * given list
	 * 
	 * @param ts the given values
	 * @return the IN predicate for the in list test
	 */
	Predicate in(List<? extends T> ts);

	/**
	 * Creates an IN predicate that tests if this value is equal to any value in a
	 * given list
	 * 
	 * @param ts the given values
	 * @return the IN predicate for the in list test
	 */
	Predicate in(@SuppressWarnings("unchecked") Value<T>... ts);

	/**
	 * Creates an IN predicate that tests if this value is equal to any value in a
	 * given list
	 * 
	 * @param ts the given values
	 * @return the IN predicate for the in list test
	 */
	Predicate in(Iterable<Value<? extends T>> ts);

	Predicate plain(Object... p);

	/**
	 * Creates a predicate that applies a given function to this values and optional
	 * additional values
	 * 
	 * @param functionName the function name
	 * @param additional   the additional values
	 * @return the function call
	 */
	Predicate func(String functionName, Value<?>... additional);

	/**
	 * Creates a predicate that tests whether this string value contains a given
	 * substring
	 * 
	 * @param substring the substring
	 * @return the predicate for the containment test
	 */
	Predicate contains(String substring);

	/**
	 * Creates a predicate that tests whether this string value contains a given
	 * substring
	 * 
	 * @param substring the substring
	 * @return the predicate for the containment test
	 */
	Predicate contains(Value<String> substring);

	/**
	 * Creates a predicate that tests whether this string value contains a given
	 * substring
	 * 
	 * @param substring       the substring
	 * @param caseInsensitive whether the test should be case insensitive
	 * @return the predicate for the containment test
	 */
	Predicate contains(Value<String> substring, boolean caseInsensitive);

	/**
	 * Creates a predicate that tests whether this string value starts with a given
	 * prefix
	 * 
	 * @param prefix the prefix
	 * @return the predicate for the starts-with test
	 */
	Predicate startsWith(String prefix);

	/**
	 * Creates a predicate that tests whether this string value starts with a given
	 * prefix
	 * 
	 * @param prefix the prefix
	 * @return the predicate for the starts-with test
	 */
	Predicate startsWith(Value<String> prefix);

	/**
	 * Creates a predicate that tests whether this string value ends with a given
	 * suffix
	 * 
	 * @param suffix the suffix
	 * @return the predicate for the ends-with test
	 */
	Predicate endsWith(String suffix);

	/**
	 * Creates a predicate that tests whether this string value ends with a given
	 * suffix
	 * 
	 * @param suffix the suffix
	 * @return the predicate for the ends-with test
	 */
	Predicate endsWith(Value<String> suffix);

	CqnSelectListValue as(String alias);

	CqnSelectListValue withoutAlias();

	<U> Value<U> type(Class<U> javaType);

	<U> Value<U> type(String cdsType);

	<U> Value<U> type(CdsBaseType cdsType);

	@Deprecated
	default <U> Value<U> cast(Class<U> javaType) {
		return type(javaType);
	}

	@Deprecated
	default <U> Value<U> cast(String cdsType) {
		return type(cdsType);
	}

}
